/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.upload;

import java.io.File;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.google.common.base.Supplier;
import com.google.inject.Singleton;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
@Singleton
class UploadServerGuice implements UploadServer {

  private final ConcurrentMap<Object, AbstractUploadScript> map = new ConcurrentHashMap<Object, AbstractUploadScript>();

  private final ExecutorService executor = Executors.newCachedThreadPool();

  @Override
  public <T> void submitSingle(UploadCtx ctx, UploadSingleAsync<T> async, T pojo, File file) {
    Object key = async.getKey();
    submit(key, new ScriptSingleSupplier<T>(ctx, async, pojo, file));
  }

  @Override
  public <T> void submitUnzip(UploadCtx ctx, UploadUnzipAsync<T> async, T pojo, UploadedZip zip) {
    Object key = async.getKey();
    submit(key, new ScriptUnzipSupplier<T>(ctx, async, pojo, zip));
  }

  @Override
  public boolean isRunning(Object key) {
    return map.containsKey(key);
  }

  private void submit(Object key, Supplier<AbstractUploadScript> supplier) {
    AbstractUploadScript script = map.get(key);

    if (script == null) {

      AbstractUploadScript newScript = supplier.get();
      script = map.putIfAbsent(key, newScript);

      if (script == null) {
        script = newScript;
      }

    }

    script.sumit();
  }

  private class ScriptSingleSupplier<T> implements Supplier<AbstractUploadScript> {

    private final UploadCtx ctx;

    private final UploadSingleAsync<T> async;

    private final T pojo;

    private final File file;

    public ScriptSingleSupplier(UploadCtx ctx, UploadSingleAsync<T> async, T pojo, File file) {
      this.ctx = ctx;
      this.async = async;
      this.pojo = pojo;
      this.file = file;
    }

    @Override
    public AbstractUploadScript get() {
      UploadedRequest request = ctx.getRequest();
      return new UploadSingleScript<T>(
          executor,
          map,
          async,
          pojo,
          file,
          request);
    }

  }

  private class ScriptUnzipSupplier<T> implements Supplier<AbstractUploadScript> {

    private final UploadCtx ctx;

    private final UploadUnzipAsync<T> async;

    private final T pojo;

    private final UploadedZip zip;

    public ScriptUnzipSupplier(UploadCtx ctx, UploadUnzipAsync<T> async, T pojo, UploadedZip zip) {
      this.ctx = ctx;
      this.async = async;
      this.pojo = pojo;
      this.zip = zip;
    }

    @Override
    public AbstractUploadScript get() {
      UploadedRequest request = ctx.getRequest();
      return new UploadUnzipScript<T>(
          executor,
          map,
          async,
          pojo,
          zip,
          request);
    }
  }

}