/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.upload;

import static com.google.common.collect.Lists.newArrayList;

import java.io.File;
import java.util.List;

import br.com.objectos.comuns.web.upload.PotentialFile;
import br.com.objectos.comuns.web.upload.UploadRequest;
import br.com.objectos.comuns.web.upload.UploadRequestException;
import br.com.objectos.comuns.web.upload.UploadRequests;
import br.com.objectos.comuns.web.upload.UploadedFile;
import br.com.objectos.comuns.web.upload.UploadedForm;
import br.com.objectos.way.ui.BaseUrl;
import br.com.objectos.way.ui.LocalDateFormat;
import br.com.objectos.way.ui.WebSession;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;

import org.joda.time.LocalDate;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
class UploadsGuice implements Uploads {

  private final BaseUrl baseUrl;
  private final UploadRequests requests;
  private final WebSession session;
  private final UploadServer uploadServer;

  @Inject
  public UploadsGuice(BaseUrl baseUrl,
                      UploadRequests requests,
                      WebSession session,
                      UploadServer uploadServer) {
    this.baseUrl = baseUrl;
    this.requests = requests;
    this.session = session;
    this.uploadServer = uploadServer;
  }

  @Override
  public FirstFromRequest getFirstFromRequest() {
    UploadRequest request = requests.get();

    Iterable<PotentialFile> potentials;
    potentials = request.getPotentialFiles();

    ToUploadedFile closure = new ToUploadedFile();

    Iterable<UploadedFile> lazyFiles;
    lazyFiles = Iterables.transform(potentials, closure);

    List<UploadedFile> files;
    files = ImmutableList.copyOf(lazyFiles);

    boolean hasErrors = closure.hasErrors();
    boolean hasFiles = !files.isEmpty();

    UploadedForm form = request.getUploadedForm();
    UploadCtx ctx = new UploadCtx(baseUrl, form, session, uploadServer);

    if (hasFiles && !hasErrors) {
      UploadedFile file = files.get(0);
      ctx = ctx.withFile(file);
      return new FirstFromRequestValid(ctx, file);
    } else {
      List<UploadRequestException> exceptions = closure.getExceptions();
      return new FirstFromRequestInvalid(ctx, exceptions);
    }
  }

  @Override
  public ReprocessFile reprocessFile(File file) {
    UploadedForm form = EmptyUploadedForm.INSTANCE;
    UploadCtx ctx = new UploadCtx(baseUrl, form, session, uploadServer);
    return new ReprocessFileValid(ctx, file);
  }

  @Override
  public boolean isRunning(Object key) {
    return uploadServer.isRunning(key);
  }

  private class ToUploadedFile implements Function<PotentialFile, UploadedFile> {

    private final List<UploadRequestException> exceptions = newArrayList();

    @Override
    public UploadedFile apply(PotentialFile input) {
      try {
        return input.saveAndGet();
      } catch (UploadRequestException e) {
        exceptions.add(e);
        return null;
      }
    }

    public List<UploadRequestException> getExceptions() {
      return exceptions;
    }

    public boolean hasErrors() {
      return !exceptions.isEmpty();
    }

  }

  private static enum EmptyUploadedForm implements UploadedForm {
    INSTANCE;
    @Override
    public String param(String key) {
      return null;
    }

    @Override
    public boolean booleanParam(String param) {
      return false;
    }

    @Override
    public <E extends Enum<E>> E enumParam(Class<E> enumClass, String param) {
      return null;
    }

    @Override
    public LocalDate localDateParam(String param) {
      return null;
    }

    @Override
    public LocalDate localDateParam(LocalDateFormat format, String param) {
      return null;
    }

    @Override
    public Double doubleParam(String param) {
      return null;
    }

    @Override
    public Integer integerParam(String param) {
      return null;
    }

    @Override
    public Long longParam(String param) {
      return null;
    }
  }

}