/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.upload.async;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;

import org.joda.time.DateTime;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class AsyncFiles {

  private AsyncFiles() {
  }

  public static List<File> listFiles(File dir) {
    return listFilesWithExt(dir, "bin");
  }
  public static List<File> listFilesWithExt(File dir, String ext) {
    Preconditions.checkArgument(dir.isDirectory(), "dir must be a directory");
    File[] _files = dir.listFiles(new ExtFilter(ext));
    List<File> files = newArrayList(_files);
    Collections.sort(files, Sort.INSTANCE);
    return ImmutableList.copyOf(files);
  }

  public static List<AsyncFile> listAsyncFiles(File dir) {
    return listAsyncFilesWithExt(dir, "bin");
  }
  public static List<AsyncFile> listAsyncFilesWithExt(File dir, String ext) {
    List<File> files = listFilesWithExt(dir, ext);
    List<AsyncFile> asyncs = transform(files, new ToAsyncFile());
    return ImmutableList.copyOf(asyncs);
  }

  public static AsyncFile of(File file) {
    return new Builder(file).build();
  }

  private static class ExtFilter implements FilenameFilter {

    private final String ext;

    public ExtFilter(String ext) {
      this.ext = "." + ext;
    }

    @Override
    public boolean accept(File dir, String name) {
      return name.endsWith(ext);
    }

  }

  private static enum Sort implements Comparator<File> {
    INSTANCE;
    @Override
    public int compare(File o1, File o2) {
      return ComparisonChain.start()
          .compare(o2.lastModified(), o1.lastModified())
          .result();
    }
  }

  private static class ToAsyncFile implements Function<File, AsyncFile> {
    @Override
    public AsyncFile apply(File input) {
      return new Builder(input).build();
    }
  }

  private static class Builder implements AsyncFile.Builder {

    private final File file;

    public Builder(File file) {
      this.file = file;
    }

    @Override
    public AsyncFile build() {
      return new AsyncFilePojo(this);
    }

    @Override
    public String getName() {
      return file.getName();
    }

    @Override
    public DateTime getDateTime() {
      long lastModified = file.lastModified();
      return new DateTime(lastModified);
    }

    @Override
    public AsyncFileStatus getStatus() {
      return AsyncFileStatus.read(file);
    }

  }

}