package buzz.getcoco.media;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * The singleton for interacting with the SDK with
 * Commands, Responses and Listeners for the corresponding commands.
 */
public class CocoMediaClient {

  static CocoMediaClient instance;

  private final Gson gson;

  private final AuthListener authListener;
  private final NativeInterface nativeHandler;
  private final NativeCallbacksInterface callbackHandler;

  // todo: add network map for getAllNetworks purpose.
  // NOTE: Intentionally not maintaining the network map. Will add if needed.

  private CocoMediaClient(AuthListener authListener,
                          NativeInterface nativeHandler,
                          NativeCallbacksInterface callbackHandler,
                          String cwdPath, String appAccessList,
                          String clientId, String downloadPath) {

    synchronized (CocoMediaClient.class) {
      if (null != CocoMediaClient.instance) {
        throw new IllegalArgumentException("CocoMediaClient already initialized");
      }

      CocoMediaClient.instance = this;
    }

    GsonBuilder builder = new GsonBuilder();

    Command.init(builder);
    Network.init(builder);
    CocoMediaClient.init(builder);

    this.gson = builder.create();
    this.authListener = authListener;
    this.nativeHandler = nativeHandler;
    this.callbackHandler = callbackHandler;

    getNativeHandler().init(cwdPath, appAccessList, clientId, downloadPath);
  }

  static void init(GsonBuilder builder) {
    builder.registerTypeAdapter(CommandId.class,
        (JsonSerializer<CommandId>) (src, typeOfSrc, context) -> new JsonPrimitive(src.getInt()));
  }

  NativeInterface getNativeHandler() {
    return nativeHandler;
  }

  AuthListener getAuthListener() {
    return authListener;
  }

  NativeCallbacksInterface getCallbackHandler() {
    return callbackHandler;
  }

  Gson getGson() {
    return gson;
  }

  /**
   * Get the singleton Object of {@link CocoMediaClient}.
   *
   * @return The instance instantiated using {@link Configurator#configure()}.
   */
  public static CocoMediaClient getInstance() {
    return instance;
  }

  /**
   * Set the access tokens which will be used in the upcoming api calls.
   *
   * @param tokens The access token json which was gotten using login HTTP apis.
   */
  public void setTokens(String tokens) {
    getNativeHandler().setTokens(tokens);
  }

  /**
   * Send the commands specified by {@link CommandId}.
   *
   * @param command  The command which will be executed
   * @param listener The Listener which will be invoked with
   *                 the status after the command is executed
   * @param <V>      The capture for {@link CommandResponse}
   * @param <U>      The capture for {@link Command}
   */
  public <V extends CommandResponse<CommandId>, U extends Command<CommandId, V>>
      void sendCommand(U command, CommandStatusListener<V> listener) {

    getNativeHandler().sendHttpCommand(command, listener);
  }

  @Override
  public String toString() {
    return "CocoMediaClient{"
           + "authListener=" + authListener
           + ", nativeHandler=" + nativeHandler
           + ", callbackHandler=" + callbackHandler
           + '}';
  }

  /**
   * The configuration builder used to initialize the SDK.
   */
  public static final class Configurator {

    private AuthListener authListener;
    private NativeInterface nativeHandler;
    private NativeCallbacksInterface callbackHandler;

    private String cwdPath;
    private String appAccessList;
    private String clientId;
    private String downloadPath;

    public Configurator() {}

    /**
     * Sets the current working directory for the SDK to use.
     *
     * @param cwdPath Path to the folder which will be used for storing persistent data
     * @return this
     */
    public Configurator setCurrentWorkingDirectory(String cwdPath) {
      this.cwdPath = cwdPath;
      return this;
    }

    /**
     * Sets the capabilities which will be used by the app.
     *
     * @param appAccessList The json with all required capabilities
     *                      eg: {"appCapabilities": [0]}
     * @return this
     */
    public Configurator setAppAccessList(String appAccessList) {
      this.appAccessList = appAccessList;
      return this;
    }

    /**
     * Sets the unique client id generated by the manage coco portal.
     *
     * @param clientId The unique string assigned to this app
     * @return this
     */
    public Configurator setClientId(String clientId) {
      this.clientId = clientId;
      return this;
    }

    /**
     * Sets the temporary folder used by SDK.
     *
     * @param downloadPath the location to the folder
     * @return this
     */
    public Configurator setDownloadDirectory(String downloadPath) {
      this.downloadPath = downloadPath;
      return this;
    }

    /**
     * Sets the NativeHandler (use this for TESTING purposes only).
     *
     * @param nativeHandler The handler to which all further calls will be forwarded to
     * @return this
     */
    public Configurator setHandler(NativeInterface nativeHandler) {
      this.nativeHandler = nativeHandler;
      return this;
    }

    /**
     * Sets the CallbackHandler (use this for TESTING purposes only).
     *
     * @param callbackHandler The handler to which all further callbacks will be forwarded to
     * @return this
     */
    public Configurator setCallbackHandler(NativeCallbacksInterface callbackHandler) {
      this.callbackHandler = callbackHandler;
      return this;
    }

    /**
     * Sets the listener for authentication.
     *
     * @param authListener The listener which will be triggered when the tokens expire
     * @return this
     */
    public Configurator setAuthListener(AuthListener authListener) {
      this.authListener = authListener;
      return this;
    }

    /**
     * Finalize the properties and initialize the SDK.
     *
     * @return The configured {@link CocoMediaClient}
     */
    public synchronized CocoMediaClient configure() {
      Objects.requireNonNull(authListener);
      Objects.requireNonNull(cwdPath);
      Objects.requireNonNull(clientId);

      if (null == appAccessList) {
        appAccessList = "{\"appCapabilities\": [0]}";
      }

      if (null == downloadPath) {
        downloadPath = cwdPath;
      }

      if (null == nativeHandler) {
        nativeHandler = new DefaultNativeHandler();
      }

      if (null == callbackHandler) {
        callbackHandler = new DefaultNativeCallbacksHandler();
      }

      // NOTE: this constructor will throw exception and fail, if called for second time.
      return new CocoMediaClient(authListener, nativeHandler,
          callbackHandler, cwdPath, appAccessList, clientId, downloadPath);
    }

    @Override
    public String toString() {
      return "Configurator{"
             + "authListener=" + authListener
             + ", nativeHandler=" + nativeHandler
             + ", callbackHandler=" + callbackHandler
             + ", cwdPath='" + cwdPath + '\''
             + ", appAccessList='" + appAccessList + '\''
             + ", clientId='" + clientId + '\''
             + ", downloadPath='" + downloadPath + '\''
             + '}';
    }
  }

  /**
   * An enum representing the possible commands of
   * {@link #sendCommand(Command, CommandStatusListener)}.
   */
  public enum CommandId implements CommandIdInterface {
    GET_ALL_NETWORKS,
    CREATE_NETWORK,
    DELETE_NETWORK,
    GET_USERS,
    REMOVE_USER,
    INVITE_USER,
    INVITE_EXTERNAL_USER;

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  /**
   * The command for getting the Networks of the current user.
   * The user will be identified by the token passed using {@link #setTokens(String)}
   */
  public static class GetAllNetworks extends Command<CommandId, GetAllNetworksResponse> {

    public GetAllNetworks() {
      super(CommandId.GET_ALL_NETWORKS, GetAllNetworksResponse.class);
    }
  }

  /**
   * Response to the {@link GetAllNetworks} command.
   */
  public static class GetAllNetworksResponse extends CommandResponse<CommandId> {

    @SerializedName(Constants.NETWORKS)
    private final ArrayList<Network> networks;

    public GetAllNetworksResponse(ArrayList<Network> networks) {
      super(CommandId.GET_ALL_NETWORKS);
      this.networks = networks;
    }

    public List<Network> getNetworks() {
      return networks;
    }

    @Override
    public String toString() {
      return "GetAllNetworksResponse{"
             + "networks=" + networks
             + "} "
             + super.toString();
    }
  }

  /**
   * The command for creating a network in the current user's id.
   * The user will be identified using the token passed with {@link #setTokens(String)}
   */
  public static class CreateNetwork extends Command<CommandId, CreateNetworkResponse> {

    @SerializedName(Constants.NETWORK_METADATA)
    private final String networkMetadata;

    @SerializedName(Constants.NETWORK_NAME)
    private final String networkName;

    @SerializedName(Constants.NETWORK_TYPE)
    private final Network.Type networkType;

    /**
     * The constructor for this class.
     *
     * @param networkName     The name of the network
     * @param networkMetadata The metadata of the network
     */
    public CreateNetwork(String networkName,
                         String networkMetadata) {

      super(CommandId.CREATE_NETWORK, CreateNetworkResponse.class);

      this.networkMetadata = networkMetadata;
      this.networkName = networkName;
      this.networkType = Network.Type.MEDIA_NET;
    }

    @Override
    public String toString() {
      return "CreateNetwork{"
             + "networkMetadata='" + networkMetadata + '\''
             + ", networkName='" + networkName + '\''
             + ", networkType=" + networkType
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link CreateNetwork} command.
   */
  public static class CreateNetworkResponse extends CommandResponse<CommandId> {
    @SerializedName(Constants.NETWORK_ID)
    private final String networkId;

    public CreateNetworkResponse(String networkId) {
      super(CommandId.CREATE_NETWORK);
      this.networkId = networkId;
    }

    public String getNetworkId() {
      return networkId;
    }

    @Override
    public String toString() {
      return "CreateNetworkResponse{"
             + "networkId='" + networkId + '\''
             + "} "
             + super.toString();
    }
  }

  /**
   * The command for deleting a network of the current user.
   * The user will be identified using the token passed with {@link #setTokens(String)}
   */
  public static class DeleteNetwork extends Command<CommandId, DeleteNetworkResponse> {
    @SerializedName(Constants.NETWORK_ID)
    private final String networkId;

    /**
     * The constructor for this class.
     *
     * @param networkId The id of the network which has to be deleted
     */
    public DeleteNetwork(String networkId) {
      super(CommandId.DELETE_NETWORK, DeleteNetworkResponse.class);
      this.networkId = networkId;
    }

    @Override
    public String toString() {
      return "DeleteNetwork{"
             + "networkId='" + networkId + '\''
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link DeleteNetwork} command.
   */
  public static class DeleteNetworkResponse extends CommandResponse<CommandId> {
    public DeleteNetworkResponse() {
      super(CommandId.DELETE_NETWORK);
    }
  }

  /**
   * The command for getting the users of a network.
   */
  public static class GetUsers extends Command<CommandId, GetUsersResponse> {

    @SerializedName(Constants.NETWORK_ID)
    private final String networkId;

    /**
     * The constructor for this class.
     *
     * @param networkId The id of the network for which the users have to be retrieved.
     */
    public GetUsers(String networkId) {
      super(CommandId.GET_USERS, GetUsersResponse.class);
      this.networkId = networkId;
    }

    @Override
    public String toString() {
      return "GetUsers{"
             + "networkId='" + networkId + '\''
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link GetUsers} command.
   */
  public static class GetUsersResponse extends CommandResponse<CommandId> {

    @SerializedName(Constants.NETWORK_USERS)
    private final ArrayList<User> users;

    public GetUsersResponse(ArrayList<User> users) {
      super(CommandId.GET_USERS);
      this.users = users;
    }

    public List<User> getUsers() {
      return users;
    }

    @Override
    public String toString() {
      return "GetUsersResponse{"
             + "users=" + users
             + "} "
             + super.toString();
    }
  }

  /**
   * Remove a user from the specified network.
   */
  public static class RemoveUser extends Command<CommandId, RemoveUserResponse> {

    @SerializedName(Constants.NETWORK_ID)
    private final String networkId;
    @SerializedName(Constants.USER_ID)
    private final String userId;

    /**
     * The constructor for this class.
     *
     * @param networkId The id of the network from which the user has to be removed.
     * @param userId    The id of the user who will to be removed.
     */
    public RemoveUser(String networkId, String userId) {
      super(CommandId.REMOVE_USER, RemoveUserResponse.class);

      this.networkId = networkId;
      this.userId = userId;
    }

    @Override
    public String toString() {
      return "RemoveUser{"
             + "networkId='" + networkId + '\''
             + ", userId='" + userId + '\''
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link RemoveUser} command.
   */
  public static class RemoveUserResponse extends CommandResponse<CommandId> {

    public RemoveUserResponse() {
      super(CommandId.REMOVE_USER);
    }
  }

  /**
   * Invite a user into the specified network.
   */
  public static class InviteUser extends Command<CommandId, InviteUserResponse> {

    @SerializedName(Constants.NETWORK_ID)
    private final String networkId;
    @SerializedName(Constants.USERNAME)
    private final String username;
    @SerializedName(Constants.ROLE)
    private final Network.Role role;
    @SerializedName(Constants.ACCESS_TYPE)
    private final Network.AccessType accessType;

    /**
     * The constructor for this class.
     *
     * @param networkId  The id of the network to which the user has to be invited.
     * @param username   The email of the user who has to be invited.
     * @param role       The role which the invited user will possess
     * @param accessType The accessType which the invited user will possess
     */
    public InviteUser(String networkId, String username,
                      Network.Role role, Network.AccessType accessType) {

      super(CommandId.INVITE_USER, InviteUserResponse.class);

      this.networkId = networkId;
      this.username = username;
      this.role = role;
      this.accessType = accessType;
    }

    @Override
    public String toString() {
      return "InviteUser{"
             + "networkId='" + networkId + '\''
             + ", username='" + username + '\''
             + ", role=" + role
             + ", accessType=" + accessType
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link InviteUser} command.
   */
  public static class InviteUserResponse extends CommandResponse<CommandId> {

    public InviteUserResponse() {
      super(CommandId.INVITE_USER);
    }
  }

  /**
   * Invite an external user into the specified network.
   */
  public static class InviteExternalUser extends Command<CommandId, InviteExternalUserResponse> {

    @SerializedName(Constants.NETWORK_ID)
    private final String networkId;
    @SerializedName(Constants.USER_ID)
    private final String userId;

    /**
     * The constructor for this class.
     *
     * @param networkId The id of the network to which the user has to be invited
     * @param userId    The <b>EXTERNAL</b> userId of the user
     */
    public InviteExternalUser(String networkId, String userId) {
      super(CommandId.INVITE_EXTERNAL_USER, InviteExternalUserResponse.class);
      this.networkId = networkId;
      this.userId = userId;
    }

    @Override
    public String toString() {
      return "InviteExternalUser{"
             + "networkId='" + networkId + '\''
             + ", userId='" + userId + '\''
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link InviteExternalUser} command.
   */
  public static class InviteExternalUserResponse extends CommandResponse<CommandId> {

    public InviteExternalUserResponse() {
      super(CommandId.INVITE_EXTERNAL_USER);
    }
  }

  /**
   * The listener for status updates of sent commands
   * {@link #sendCommand(Command, CommandStatusListener)}.
   *
   * @param <T> The capture for {@link CommandResponse}
   */
  public interface CommandStatusListener<T extends CommandResponse<CommandId>> extends Listener {
    void onResponse(T response, Throwable tr);
  }

  /**
   * The listener for authentication. Triggered when the tokens set using
   * {@link #setTokens(String)} are expired or not set at all.
   */
  public interface AuthListener {
    void onAuthRequired(String authEndPoint, String tokenEndPoint);
  }
}
