package buzz.getcoco.media;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

class DefaultNativeCallbacksHandler implements NativeCallbacksInterface {

  private static final Logger LOGGER = Util.getLogger();

  private <T extends CommandIdInterface>
      Throwable getErrorFromResponse(CommandResponse<T> response) {

    if (null == response) {
      return new IllegalStateException("unable to parse response");
    }

    if (Command.State.SUCCESS == response.getStatus()) {
      return null;
    }

    return new IllegalStateException("failure response: " + response);
  }

  @Override
  public void authCallback(String authEndPoint, String tokenEndPoint) {
    CocoMediaClient.getInstance().getAuthListener().onAuthRequired(authEndPoint, tokenEndPoint);
    LOGGER.log(Level.FINE, "authCallback: authEndPoint: " + authEndPoint
                           + ", tokenEndPoint: " + tokenEndPoint);
  }

  @Override
  public void httpCommandStatusCallback(String response, Object context) {
    LOGGER.fine("response: " + response + ", context: " + context);

    Context castedContext = (Context) context;

    Command<CocoMediaClient.CommandId, CommandResponse<CocoMediaClient.CommandId>> command =
        castedContext.getSdkContext();

    CocoMediaClient.CommandStatusListener<CommandResponse<CocoMediaClient.CommandId>> listener =
        castedContext.getDeveloperContext();

    CommandResponse<CocoMediaClient.CommandId> commandResponse =
        command.createResponse(response);

    listener.onResponse(commandResponse, getErrorFromResponse(commandResponse));

    LOGGER.log(Level.FINE, "response: " + response);
  }

  @Override
  public void networkCommandStatusCallback(String response, Object context) {
    LOGGER.fine("response: " + response + ", context: " + context);

    Context castedContext = (Context) context;

    Command<Network.CommandId, CommandResponse<Network.CommandId>> command =
        castedContext.getSdkContext();

    Network.CommandStatusListener<CommandResponse<Network.CommandId>> listener =
        castedContext.getDeveloperContext();

    CommandResponse<Network.CommandId> commandResponse =
        command.createResponse(response);

    listener.onResponse(commandResponse, getErrorFromResponse(commandResponse));

    LOGGER.log(Level.FINE, "response: " + response);
  }

  @Override
  public void connectStatusCallback(int status, Object connectContext) {
    LOGGER.fine("status: " + status + ", context: " + connectContext);

    Context castedContext = (Context) connectContext;

    Network network = castedContext.getSdkContext();
    Network.State networkStatus = Network.State.getValue(status);

    network.internalSetStatus(networkStatus);

    LOGGER.log(Level.FINE, "network: " + network + ", state: " + status);
  }

  @Override
  public void channelJoinStatusCallback(int channelId, int status, Object joinContext) {
    LOGGER.fine("channelId: " + channelId + ", status: " + status + ", context: " + joinContext);

    Context castedContext = (Context) joinContext;

    Channel channel = castedContext.getSdkContext();
    Channel.ChannelStatusListener listener = castedContext.getDeveloperContext();
    Channel.State joinStatus = Channel.State.getValue(status);

    channel.internalSetState(joinStatus);

    listener.onJoinStatusChanged(joinStatus);

    LOGGER.log(Level.FINE, "channel: " + channel);
  }

  @Override
  public void channelStreamInfoCallback(int channelId, long streamId, long sourceNodeId,
                                        int status, String sdp, Object joinContext) {
    LOGGER.fine("channelId: " + channelId + ", streamId: " + streamId + ", nodeId: " + sourceNodeId
                + ", status: " + status + ", joinContext: " + joinContext);

    Context castedContext = (Context) joinContext;

    Channel channel = castedContext.getSdkContext();
    Channel.ChannelStatusListener listener = castedContext.getDeveloperContext();

    Objects.requireNonNull(listener);

    RxStreamInfo streamInfo =
        new RxStreamInfo(sdp, channel.getNetworkId(), channelId, streamId, sourceNodeId);

    Stream.State streamStatus = Stream.State.getValue(status, false);

    if (Stream.State.CREATED == streamStatus) {
      RxStreamInfo.activateStream(streamInfo);
    }

    if (Stream.State.DESTROYED == streamStatus) {
      RxStreamInfo.deactivateStream(streamInfo);
    }

    listener.onStreamInfoUpdated(streamInfo, streamStatus);
  }

  @Override
  public void streamStatusChangedCallback(int channelId, long streamHandle, long sourceNodeId,
                                          int status, Object streamContext) {

    LOGGER.fine("channelId: " + channelId + ", streamHandle: " + streamHandle
                + ", status: " + status + ", streamContext: " + streamContext);

    Context castedContext = (Context) streamContext;

    Stream stream = castedContext.getSdkContext();
    Stream.StreamStatusListener listener = castedContext.getDeveloperContext();
    Stream.State streamStatus = Stream.State.getValue(status, stream.isTxStream());

    streamHandle = (Stream.State.STARTED == streamStatus) ? streamHandle : 0;

    Objects.requireNonNull(listener);

    stream.setStreamHandle(streamHandle);

    stream.internalSetSourceNodeId(sourceNodeId);
    stream.internalSetStatus(streamStatus);

    listener.onStreamStatusChanged(streamStatus);

    LOGGER.log(Level.FINE, "stream: " + stream);

    // The streamHandle will be freed after this call returns.
    // Any existing send / close calls MUST be completed before that.
    if (0 == streamHandle) {
      stream.waitUntilAllSendersReturn();
    }
  }

  @Override
  public void streamDataReceivedCallback(ByteBuffer data, Object streamContext) {

    Context castedContext = (Context) streamContext;

    Stream.StreamStatusListener listener = castedContext.getDeveloperContext();

    data.order(ByteOrder.nativeOrder());

    listener.onStreamDataReceived(new Frame(data));
  }

  @Override
  public void messageReceivedCallback(String message, long sourceNodeId, Object connectContext) {
    LOGGER.fine("message received: " + message + ", source: " + sourceNodeId);
    LOGGER.fine("context: " + connectContext);

    Context castedContext = (Context) connectContext;

    Network network = castedContext.getSdkContext();

    network.internalOnMessageReceived(message, sourceNodeId);
  }

  @Override
  public void contentInfoReceivedCallback(String message, int contentTime,
                                          long sourceNodeId, Object connectContext) {
    LOGGER.fine("message received: " + message + ", source: " + sourceNodeId);
    LOGGER.fine("context: " + connectContext);

    Context castedContext = (Context) connectContext;

    Network network = castedContext.getSdkContext();

    network.internalOnContentInfoMessageReceived(message, sourceNodeId, contentTime);
  }

  @Override
  public void nodeConnectionStatusCallback(long nodeId, boolean isOnline, Object connectContext) {
    LOGGER.fine("node: " + nodeId + ", isOnline: " + isOnline);
    LOGGER.fine("context: " + connectContext);

    Context castedContext = (Context) connectContext;

    Network network = castedContext.getSdkContext();

    network.internalNodeConnectionStatusCallback(nodeId, isOnline);
  }

  @Override
  public String toString() {
    return "DefaultNativeCallbacksHandler{}";
  }
}
