package ca.uhn.fhir.model.primitive;

/*
 * #%L
 * HAPI FHIR Library
 * %%
 * Copyright (C) 2014 University Health Network
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;

import ca.uhn.fhir.model.api.BasePrimitive;
import ca.uhn.fhir.model.api.annotation.DatatypeDef;
import ca.uhn.fhir.model.api.annotation.SimpleSetter;
import ca.uhn.fhir.parser.DataFormatException;

@DatatypeDef(name = "decimal")
public class DecimalDt extends BasePrimitive<BigDecimal> implements Comparable<DecimalDt> {

	private BigDecimal myValue;

	/**
	 * Constructor
	 */
	public DecimalDt() {
		super();
	}

	public Number getValueAsNumber() {
		return myValue; 
	}
	
	/**
	 * Constructor
	 */
	@SimpleSetter
	public DecimalDt(@SimpleSetter.Parameter(name = "theValue") BigDecimal theValue) {
		setValue(theValue);
	}

	/**
	 * Constructor
	 */
	@SimpleSetter
	public DecimalDt(@SimpleSetter.Parameter(name = "theValue") double theValue) {
		// Use the valueOf here because the constructor gives crazy precision
		// changes due to the floating point conversion
		setValue(BigDecimal.valueOf(theValue));
	}

	/**
	 * Rounds the value to the given prevision
	 * 
	 * @see MathContext#getPrecision()
	 */
	public void round(int thePrecision) {
		if (getValue()!=null) {
			BigDecimal newValue = getValue().round(new MathContext(thePrecision));
			setValue(newValue);
		}
	}

	/**
	 * Rounds the value to the given prevision
	 * 
	 * @see MathContext#getPrecision()
	 * @see MathContext#getRoundingMode()
	 */
	public void round(int thePrecision, RoundingMode theRoundingMode) {
		if (getValue()!=null) {
			BigDecimal newValue = getValue().round(new MathContext(thePrecision, theRoundingMode));
			setValue(newValue);
		}
	}

	/**
	 * Constructor
	 */
	@SimpleSetter
	public DecimalDt(@SimpleSetter.Parameter(name = "theValue") long theValue) {
		setValue(new BigDecimal(theValue));
	}

	/**
	 * Constructor
	 */
	public DecimalDt(String theValue) {
		setValue(new BigDecimal(theValue));
	}

	@Override
	public void setValueAsString(String theValue) throws DataFormatException {
		if (theValue == null) {
			myValue = null;
		} else {
			myValue = new BigDecimal(theValue);
		}
	}

	@Override
	public String getValueAsString() {
		if (myValue == null) {
			return null;
		}
		return myValue.toPlainString();
	}

	@Override
	public BigDecimal getValue() {
		return myValue;
	}

	@Override
	public void setValue(BigDecimal theValue) throws DataFormatException {
		myValue = theValue;
	}

	/**
	 * Sets a new value using an integer
	 */
	public void setValueAsInteger(int theValue) {
		myValue = new BigDecimal(theValue);
	}

	/**
	 * Gets the value as an integer, using {@link BigDecimal#intValue()}
	 */
	public int getValueAsInteger() {
		return myValue.intValue();
	}

	@Override
	public int compareTo(DecimalDt theObj) {
		if (myValue == null && theObj.getValue() == null) {
			return 0;
		}
		if (myValue != null && theObj.getValue() == null) {
			return 1;
		}
		if (myValue == null && theObj.getValue() != null) {
			return -1;
		}
		return myValue.compareTo(theObj.getValue());
	}

}
