/*
 * #%L
 * HAPI FHIR - Client Framework
 * %%
 * Copyright (C) 2014 - 2025 Smile CDR, Inc.
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package ca.uhn.fhir.rest.client.method;

import ca.uhn.fhir.context.ConfigurationException;
import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.i18n.Msg;
import ca.uhn.fhir.model.primitive.IntegerDt;
import ca.uhn.fhir.rest.annotation.Count;
import ca.uhn.fhir.rest.api.Constants;
import ca.uhn.fhir.rest.param.ParameterUtil;
import ca.uhn.fhir.rest.server.exceptions.InternalErrorException;
import org.hl7.fhir.instance.model.api.IBaseResource;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class CountParameter implements IParameter {

	@Override
	public void translateClientArgumentIntoQueryArgument(
			FhirContext theContext,
			Object theSourceClientArgument,
			Map<String, List<String>> theTargetQueryArguments,
			IBaseResource theTargetResource)
			throws InternalErrorException {
		if (theSourceClientArgument != null) {
			IntegerDt since = ParameterUtil.toInteger(theSourceClientArgument);
			if (since.isEmpty() == false) {
				theTargetQueryArguments.put(Constants.PARAM_COUNT, Collections.singletonList(since.getValueAsString()));
			}
		}
	}

	@Override
	public void initializeTypes(
			Method theMethod,
			Class<? extends Collection<?>> theOuterCollectionType,
			Class<? extends Collection<?>> theInnerCollectionType,
			Class<?> theParameterType) {
		if (theOuterCollectionType != null) {
			throw new ConfigurationException(Msg.code(1420) + "Method '" + theMethod.getName() + "' in type '"
					+ theMethod.getDeclaringClass().getCanonicalName() + "' is annotated with @" + Count.class.getName()
					+ " but can not be of collection type");
		}
		if (!ParameterUtil.isBindableIntegerType(theParameterType)) {
			throw new ConfigurationException(Msg.code(1421) + "Method '" + theMethod.getName() + "' in type '"
					+ theMethod.getDeclaringClass().getCanonicalName() + "' is annotated with @" + Count.class.getName()
					+ " but type '" + theParameterType + "' is an invalid type, must be Integer or IntegerType");
		}
	}
}
