/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.fhir.jpa.bulk.imprt.svc;

import ca.uhn.fhir.jpa.api.config.DaoConfig;
import ca.uhn.fhir.jpa.batch.api.IBatchJobSubmitter;
import ca.uhn.fhir.jpa.batch.log.Logs;
import ca.uhn.fhir.jpa.bulk.imprt.api.IBulkDataImportSvc;
import ca.uhn.fhir.jpa.bulk.imprt.model.BulkImportJobFileJson;
import ca.uhn.fhir.jpa.bulk.imprt.model.BulkImportJobJson;
import ca.uhn.fhir.jpa.bulk.imprt.model.BulkImportJobStatusEnum;
import ca.uhn.fhir.jpa.dao.data.IBulkImportJobDao;
import ca.uhn.fhir.jpa.dao.data.IBulkImportJobFileDao;
import ca.uhn.fhir.jpa.entity.BulkImportJobEntity;
import ca.uhn.fhir.jpa.entity.BulkImportJobFileEntity;
import ca.uhn.fhir.jpa.model.sched.HapiJob;
import ca.uhn.fhir.jpa.model.sched.ISchedulerService;
import ca.uhn.fhir.jpa.model.sched.ScheduledJobDefinition;
import ca.uhn.fhir.rest.server.exceptions.InvalidRequestException;
import ca.uhn.fhir.util.ValidateUtil;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Semaphore;
import javax.annotation.Nonnull;
import javax.annotation.PostConstruct;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.Job;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.JobParametersInvalidException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Slice;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

public class BulkDataImportSvcImpl
implements IBulkDataImportSvc {
    private static final Logger ourLog = LoggerFactory.getLogger(BulkDataImportSvcImpl.class);
    private final Semaphore myRunningJobSemaphore = new Semaphore(1);
    @Autowired
    private IBulkImportJobDao myJobDao;
    @Autowired
    private IBulkImportJobFileDao myJobFileDao;
    @Autowired
    private PlatformTransactionManager myTxManager;
    private TransactionTemplate myTxTemplate;
    @Autowired
    private ISchedulerService mySchedulerService;
    @Autowired
    private IBatchJobSubmitter myJobSubmitter;
    @Autowired
    @Qualifier(value="bulkImportJob")
    private Job myBulkImportJob;
    @Autowired
    private DaoConfig myDaoConfig;

    @PostConstruct
    public void start() {
        this.myTxTemplate = new TransactionTemplate(this.myTxManager);
        ScheduledJobDefinition jobDetail = new ScheduledJobDefinition();
        jobDetail.setId(ActivationJob.class.getName());
        jobDetail.setJobClass(ActivationJob.class);
        this.mySchedulerService.scheduleLocalJob(10000L, jobDetail);
    }

    @Transactional
    public String createNewJob(BulkImportJobJson theJobDescription, @Nonnull List<BulkImportJobFileJson> theInitialFiles) {
        ValidateUtil.isNotNullOrThrowUnprocessableEntity((Object)theJobDescription, (String)"Job must not be null", (Object[])new Object[0]);
        ValidateUtil.isNotNullOrThrowUnprocessableEntity((Object)theJobDescription.getProcessingMode(), (String)"Job File Processing mode must not be null", (Object[])new Object[0]);
        ValidateUtil.isTrueOrThrowInvalidRequest((theJobDescription.getBatchSize() > 0 ? 1 : 0) != 0, (String)"Job File Batch Size must be > 0", (Object[])new Object[0]);
        String jobId = UUID.randomUUID().toString();
        ourLog.info("Creating new Bulk Import job with {} files, assigning job ID: {}", (Object)theInitialFiles.size(), (Object)jobId);
        BulkImportJobEntity job = new BulkImportJobEntity();
        job.setJobId(jobId);
        job.setFileCount(theInitialFiles.size());
        job.setStatus(BulkImportJobStatusEnum.STAGING);
        job.setJobDescription(theJobDescription.getJobDescription());
        job.setBatchSize(theJobDescription.getBatchSize());
        job.setRowProcessingMode(theJobDescription.getProcessingMode());
        job = (BulkImportJobEntity)this.myJobDao.save(job);
        int nextSequence = 0;
        this.addFilesToJob(theInitialFiles, job, nextSequence);
        return jobId;
    }

    @Transactional
    public void addFilesToJob(String theJobId, List<BulkImportJobFileJson> theFiles) {
        ourLog.info("Adding {} files to bulk import job: {}", (Object)theFiles.size(), (Object)theJobId);
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        ValidateUtil.isTrueOrThrowInvalidRequest((job.getStatus() == BulkImportJobStatusEnum.STAGING ? 1 : 0) != 0, (String)"Job %s has status %s and can not be added to", (Object[])new Object[]{theJobId, job.getStatus()});
        this.addFilesToJob(theFiles, job, job.getFileCount());
        job.setFileCount(job.getFileCount() + theFiles.size());
        this.myJobDao.save(job);
    }

    private BulkImportJobEntity findJobByJobId(String theJobId) {
        BulkImportJobEntity job = this.myJobDao.findByJobId(theJobId).orElseThrow(() -> new InvalidRequestException("Unknown job ID: " + theJobId));
        return job;
    }

    @Transactional
    public void markJobAsReadyForActivation(String theJobId) {
        ourLog.info("Activating bulk import job {}", (Object)theJobId);
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        ValidateUtil.isTrueOrThrowInvalidRequest((job.getStatus() == BulkImportJobStatusEnum.STAGING ? 1 : 0) != 0, (String)"Bulk import job %s can not be activated in status: %s", (Object[])new Object[]{theJobId, job.getStatus()});
        job.setStatus(BulkImportJobStatusEnum.READY);
        this.myJobDao.save(job);
    }

    @Transactional(value=Transactional.TxType.NEVER)
    public boolean activateNextReadyJob() {
        if (!this.myDaoConfig.isEnableTaskBulkImportJobExecution()) {
            Logs.getBatchTroubleshootingLog().trace("Bulk import job execution is not enabled on this server. No action taken.");
            return false;
        }
        if (!this.myRunningJobSemaphore.tryAcquire()) {
            Logs.getBatchTroubleshootingLog().trace("Already have a running batch job, not going to check for more");
            return false;
        }
        try {
            boolean bl = this.doActivateNextReadyJob();
            return bl;
        }
        finally {
            this.myRunningJobSemaphore.release();
        }
    }

    private boolean doActivateNextReadyJob() {
        Optional jobToProcessOpt = Objects.requireNonNull((Optional)this.myTxTemplate.execute(t -> {
            PageRequest page = PageRequest.of((int)0, (int)1);
            Slice<BulkImportJobEntity> submittedJobs = this.myJobDao.findByStatus((Pageable)page, BulkImportJobStatusEnum.READY);
            if (submittedJobs.isEmpty()) {
                return Optional.empty();
            }
            return Optional.of((BulkImportJobEntity)submittedJobs.getContent().get(0));
        }));
        if (!jobToProcessOpt.isPresent()) {
            return false;
        }
        BulkImportJobEntity bulkImportJobEntity = (BulkImportJobEntity)jobToProcessOpt.get();
        String jobUuid = bulkImportJobEntity.getJobId();
        try {
            this.processJob(bulkImportJobEntity);
        }
        catch (Exception e) {
            ourLog.error("Failure while preparing bulk export extract", (Throwable)e);
            this.myTxTemplate.execute(t -> {
                Optional<BulkImportJobEntity> submittedJobs = this.myJobDao.findByJobId(jobUuid);
                if (submittedJobs.isPresent()) {
                    BulkImportJobEntity jobEntity = submittedJobs.get();
                    jobEntity.setStatus(BulkImportJobStatusEnum.ERROR);
                    jobEntity.setStatusMessage(e.getMessage());
                    this.myJobDao.save(jobEntity);
                }
                return false;
            });
        }
        return true;
    }

    @Transactional
    public void setJobToStatus(String theJobId, BulkImportJobStatusEnum theStatus) {
        this.setJobToStatus(theJobId, theStatus, null);
    }

    public void setJobToStatus(String theJobId, BulkImportJobStatusEnum theStatus, String theStatusMessage) {
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        job.setStatus(theStatus);
        job.setStatusMessage(theStatusMessage);
        this.myJobDao.save(job);
    }

    @Transactional
    public BulkImportJobJson fetchJob(String theJobId) {
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        return job.toJson();
    }

    public IBulkDataImportSvc.JobInfo getJobStatus(String theJobId) {
        BulkImportJobEntity theJob = this.findJobByJobId(theJobId);
        return new IBulkDataImportSvc.JobInfo().setStatus(theJob.getStatus()).setStatusMessage(theJob.getStatusMessage()).setStatusTime(theJob.getStatusTime());
    }

    @Transactional
    public BulkImportJobFileJson fetchFile(String theJobId, int theFileIndex) {
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        return this.myJobFileDao.findForJob(job, theFileIndex).map(t -> t.toJson()).orElseThrow(() -> new IllegalArgumentException("Invalid index " + theFileIndex + " for job " + theJobId));
    }

    @Transactional
    public String getFileDescription(String theJobId, int theFileIndex) {
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        return this.myJobFileDao.findFileDescriptionForJob(job, theFileIndex).orElse("");
    }

    @Transactional
    public void deleteJobFiles(String theJobId) {
        BulkImportJobEntity job = this.findJobByJobId(theJobId);
        List<Long> files = this.myJobFileDao.findAllIdsForJob(theJobId);
        for (Long next : files) {
            this.myJobFileDao.deleteById(next);
        }
        this.myJobDao.delete(job);
    }

    private void processJob(BulkImportJobEntity theBulkExportJobEntity) throws JobParametersInvalidException {
        String jobId = theBulkExportJobEntity.getJobId();
        int batchSize = theBulkExportJobEntity.getBatchSize();
        ValidateUtil.isTrueOrThrowInvalidRequest((batchSize > 0 ? 1 : 0) != 0, (String)"Batch size must be positive", (Object[])new Object[0]);
        JobParametersBuilder parameters = new JobParametersBuilder().addString("jobUUID", jobId).addLong("commitInterval", Long.valueOf(batchSize));
        if (StringUtils.isNotBlank((CharSequence)theBulkExportJobEntity.getJobDescription())) {
            parameters.addString("jobDescription", theBulkExportJobEntity.getJobDescription());
        }
        ourLog.info("Submitting bulk import job {} to job scheduler", (Object)jobId);
        this.myJobSubmitter.runJob(this.myBulkImportJob, parameters.toJobParameters());
    }

    private void addFilesToJob(@Nonnull List<BulkImportJobFileJson> theInitialFiles, BulkImportJobEntity job, int nextSequence) {
        for (BulkImportJobFileJson nextFile : theInitialFiles) {
            ValidateUtil.isNotBlankOrThrowUnprocessableEntity((String)nextFile.getContents(), (String)"Job File Contents mode must not be null");
            BulkImportJobFileEntity jobFile = new BulkImportJobFileEntity();
            jobFile.setJob(job);
            jobFile.setContents(nextFile.getContents());
            jobFile.setTenantName(nextFile.getTenantName());
            jobFile.setFileDescription(nextFile.getDescription());
            jobFile.setFileSequence(nextSequence++);
            this.myJobFileDao.save(jobFile);
        }
    }

    public static class ActivationJob
    implements HapiJob {
        @Autowired
        private IBulkDataImportSvc myTarget;

        public void execute(JobExecutionContext theContext) {
            this.myTarget.activateNextReadyJob();
        }
    }
}

