/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.fhir.jpa.partition;

import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.i18n.Msg;
import ca.uhn.fhir.jpa.dao.data.IPartitionDao;
import ca.uhn.fhir.jpa.entity.PartitionEntity;
import ca.uhn.fhir.jpa.model.config.PartitionSettings;
import ca.uhn.fhir.jpa.partition.IPartitionLookupSvc;
import ca.uhn.fhir.rest.server.exceptions.InvalidRequestException;
import ca.uhn.fhir.rest.server.exceptions.MethodNotAllowedException;
import ca.uhn.fhir.rest.server.exceptions.ResourceNotFoundException;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import javax.annotation.PostConstruct;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

public class PartitionLookupSvcImpl
implements IPartitionLookupSvc {
    private static final Pattern PARTITION_NAME_VALID_PATTERN = Pattern.compile("[a-zA-Z0-9_-]+");
    private static final Logger ourLog = LoggerFactory.getLogger(PartitionLookupSvcImpl.class);
    @Autowired
    private PartitionSettings myPartitionSettings;
    @Autowired
    private PlatformTransactionManager myTxManager;
    @Autowired
    private IPartitionDao myPartitionDao;
    private LoadingCache<String, PartitionEntity> myNameToPartitionCache;
    private LoadingCache<Integer, PartitionEntity> myIdToPartitionCache;
    private TransactionTemplate myTxTemplate;
    @Autowired
    private FhirContext myFhirCtx;

    @Override
    @PostConstruct
    public void start() {
        this.myNameToPartitionCache = Caffeine.newBuilder().expireAfterWrite(1L, TimeUnit.MINUTES).build((CacheLoader)new NameToPartitionCacheLoader());
        this.myIdToPartitionCache = Caffeine.newBuilder().expireAfterWrite(1L, TimeUnit.MINUTES).build((CacheLoader)new IdToPartitionCacheLoader());
        this.myTxTemplate = new TransactionTemplate(this.myTxManager);
    }

    @Override
    public PartitionEntity getPartitionByName(String theName) {
        Validate.notBlank((CharSequence)theName, (String)"The name must not be null or blank", (Object[])new Object[0]);
        this.validateNotInUnnamedPartitionMode();
        if ("DEFAULT".equals(theName)) {
            return null;
        }
        return (PartitionEntity)this.myNameToPartitionCache.get((Object)theName);
    }

    @Override
    public PartitionEntity getPartitionById(Integer thePartitionId) {
        PartitionLookupSvcImpl.validatePartitionIdSupplied(this.myFhirCtx, thePartitionId);
        if (this.myPartitionSettings.isUnnamedPartitionMode()) {
            return new PartitionEntity().setId(thePartitionId);
        }
        if (this.myPartitionSettings.getDefaultPartitionId() != null && this.myPartitionSettings.getDefaultPartitionId().equals(thePartitionId)) {
            return new PartitionEntity().setId(thePartitionId).setName("DEFAULT");
        }
        return (PartitionEntity)this.myIdToPartitionCache.get((Object)thePartitionId);
    }

    @Override
    public void clearCaches() {
        this.myNameToPartitionCache.invalidateAll();
        this.myIdToPartitionCache.invalidateAll();
    }

    @Override
    @Transactional
    public PartitionEntity createPartition(PartitionEntity thePartition) {
        this.validateNotInUnnamedPartitionMode();
        this.validateHaveValidPartitionIdAndName(thePartition);
        this.validatePartitionNameDoesntAlreadyExist(thePartition.getName());
        ourLog.info("Creating new partition with ID {} and Name {}", (Object)thePartition.getId(), (Object)thePartition.getName());
        this.myPartitionDao.save(thePartition);
        return thePartition;
    }

    @Override
    @Transactional
    public PartitionEntity updatePartition(PartitionEntity thePartition) {
        this.validateNotInUnnamedPartitionMode();
        this.validateHaveValidPartitionIdAndName(thePartition);
        Optional existingPartitionOpt = this.myPartitionDao.findById(thePartition.getId());
        if (!existingPartitionOpt.isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "unknownPartitionId", new Object[]{thePartition.getId()});
            throw new InvalidRequestException(Msg.code((int)1307) + msg);
        }
        PartitionEntity existingPartition = (PartitionEntity)existingPartitionOpt.get();
        if (!thePartition.getName().equalsIgnoreCase(existingPartition.getName())) {
            this.validatePartitionNameDoesntAlreadyExist(thePartition.getName());
        }
        existingPartition.setName(thePartition.getName());
        existingPartition.setDescription(thePartition.getDescription());
        this.myPartitionDao.save(existingPartition);
        this.clearCaches();
        return existingPartition;
    }

    @Override
    @Transactional
    public void deletePartition(Integer thePartitionId) {
        PartitionLookupSvcImpl.validatePartitionIdSupplied(this.myFhirCtx, thePartitionId);
        this.validateNotInUnnamedPartitionMode();
        Optional partition = this.myPartitionDao.findById(thePartitionId);
        if (!partition.isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "unknownPartitionId", new Object[]{thePartitionId});
            throw new IllegalArgumentException(Msg.code((int)1308) + msg);
        }
        this.myPartitionDao.delete((PartitionEntity)partition.get());
        this.clearCaches();
    }

    @Override
    public List<PartitionEntity> listPartitions() {
        List allPartitions = this.myPartitionDao.findAll();
        return allPartitions;
    }

    private void validatePartitionNameDoesntAlreadyExist(String theName) {
        if (this.myPartitionDao.findForName(theName).isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "cantCreateDuplicatePartitionName", new Object[]{theName});
            throw new InvalidRequestException(Msg.code((int)1309) + msg);
        }
    }

    private void validateHaveValidPartitionIdAndName(PartitionEntity thePartition) {
        if (thePartition.getId() == null || StringUtils.isBlank((CharSequence)thePartition.getName())) {
            String msg = this.myFhirCtx.getLocalizer().getMessage(PartitionLookupSvcImpl.class, "missingPartitionIdOrName", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)1310) + msg);
        }
        if (thePartition.getName().equals("DEFAULT")) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "cantCreateDefaultPartition", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)1311) + msg);
        }
        if (!PARTITION_NAME_VALID_PATTERN.matcher(thePartition.getName()).matches()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "invalidName", new Object[]{thePartition.getName()});
            throw new InvalidRequestException(Msg.code((int)1312) + msg);
        }
    }

    private void validateNotInUnnamedPartitionMode() {
        if (this.myPartitionSettings.isUnnamedPartitionMode()) {
            throw new MethodNotAllowedException(Msg.code((int)1313) + "Can not invoke this operation in unnamed partition mode");
        }
    }

    public static void validatePartitionIdSupplied(FhirContext theFhirContext, Integer thePartitionId) {
        if (thePartitionId == null) {
            String msg = theFhirContext.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "noIdSupplied", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)1314) + msg);
        }
    }

    private class NameToPartitionCacheLoader
    implements CacheLoader<String, PartitionEntity> {
        private NameToPartitionCacheLoader() {
        }

        public @Nullable PartitionEntity load(@NonNull String theName) {
            return (PartitionEntity)PartitionLookupSvcImpl.this.myTxTemplate.execute(t -> PartitionLookupSvcImpl.this.myPartitionDao.findForName(theName).orElseThrow(() -> {
                String msg = PartitionLookupSvcImpl.this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "invalidName", new Object[]{theName});
                return new ResourceNotFoundException(msg);
            }));
        }
    }

    private class IdToPartitionCacheLoader
    implements CacheLoader<Integer, PartitionEntity> {
        private IdToPartitionCacheLoader() {
        }

        public @Nullable PartitionEntity load(@NonNull Integer theId) {
            return (PartitionEntity)PartitionLookupSvcImpl.this.myTxTemplate.execute(t -> (PartitionEntity)PartitionLookupSvcImpl.this.myPartitionDao.findById(theId).orElseThrow(() -> {
                String msg = PartitionLookupSvcImpl.this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "unknownPartitionId", new Object[]{theId});
                return new ResourceNotFoundException(msg);
            }));
        }
    }
}

