/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.fhir.jpa.partition;

import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.i18n.Msg;
import ca.uhn.fhir.interceptor.api.HookParams;
import ca.uhn.fhir.interceptor.api.IInterceptorService;
import ca.uhn.fhir.interceptor.api.IPointcut;
import ca.uhn.fhir.interceptor.api.Pointcut;
import ca.uhn.fhir.interceptor.model.RequestPartitionId;
import ca.uhn.fhir.jpa.dao.data.IPartitionDao;
import ca.uhn.fhir.jpa.entity.PartitionEntity;
import ca.uhn.fhir.jpa.model.config.PartitionSettings;
import ca.uhn.fhir.jpa.partition.IPartitionLookupSvc;
import ca.uhn.fhir.jpa.util.MemoryCacheService;
import ca.uhn.fhir.rest.api.server.RequestDetails;
import ca.uhn.fhir.rest.server.exceptions.InvalidRequestException;
import ca.uhn.fhir.rest.server.exceptions.MethodNotAllowedException;
import ca.uhn.fhir.rest.server.exceptions.ResourceNotFoundException;
import ca.uhn.fhir.rest.server.servlet.ServletRequestDetails;
import ca.uhn.fhir.util.ICallable;
import jakarta.annotation.Nonnull;
import jakarta.annotation.PostConstruct;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

public class PartitionLookupSvcImpl
implements IPartitionLookupSvc {
    private static final Pattern PARTITION_NAME_VALID_PATTERN = Pattern.compile("[a-zA-Z0-9_-]+");
    private static final Logger ourLog = LoggerFactory.getLogger(PartitionLookupSvcImpl.class);
    @Autowired
    private PartitionSettings myPartitionSettings;
    @Autowired
    private IInterceptorService myInterceptorService;
    @Autowired
    private IPartitionDao myPartitionDao;
    @Autowired
    private MemoryCacheService myMemoryCacheService;
    @Autowired
    private FhirContext myFhirCtx;
    @Autowired
    private PlatformTransactionManager myTxManager;
    private TransactionTemplate myTxTemplate;

    @Override
    @PostConstruct
    public void start() {
        this.myTxTemplate = new TransactionTemplate(this.myTxManager);
    }

    @Override
    public PartitionEntity getPartitionByName(String theName) {
        Validate.notBlank((CharSequence)theName, (String)"The name must not be null or blank", (Object[])new Object[0]);
        this.validateNotInUnnamedPartitionMode();
        if ("DEFAULT".equals(theName)) {
            return null;
        }
        return (PartitionEntity)this.myMemoryCacheService.get(MemoryCacheService.CacheEnum.NAME_TO_PARTITION, (Object)theName, this::lookupPartitionByName);
    }

    @Override
    public PartitionEntity getPartitionById(Integer thePartitionId) {
        PartitionLookupSvcImpl.validatePartitionIdSupplied(this.myFhirCtx, thePartitionId);
        if (this.myPartitionSettings.isUnnamedPartitionMode()) {
            return new PartitionEntity().setId(thePartitionId);
        }
        if (this.myPartitionSettings.getDefaultPartitionId() != null && this.myPartitionSettings.getDefaultPartitionId().equals(thePartitionId)) {
            return new PartitionEntity().setId(thePartitionId).setName("DEFAULT");
        }
        return (PartitionEntity)this.myMemoryCacheService.get(MemoryCacheService.CacheEnum.ID_TO_PARTITION, (Object)thePartitionId, this::lookupPartitionById);
    }

    @Override
    public void invalidateCaches() {
        this.myMemoryCacheService.invalidateCaches(new MemoryCacheService.CacheEnum[]{MemoryCacheService.CacheEnum.NAME_TO_PARTITION, MemoryCacheService.CacheEnum.ID_TO_PARTITION});
    }

    @Override
    public int generateRandomUnusedPartitionId() {
        int candidate = ThreadLocalRandom.current().nextInt(1, Integer.MAX_VALUE);
        Optional partition = this.myPartitionDao.findById(candidate);
        while (partition.isPresent()) {
            candidate = ThreadLocalRandom.current().nextInt(1, Integer.MAX_VALUE);
            partition = this.myPartitionDao.findById(candidate);
        }
        return candidate;
    }

    @Override
    @Transactional
    public PartitionEntity createPartition(PartitionEntity thePartition, RequestDetails theRequestDetails) {
        this.validateNotInUnnamedPartitionMode();
        this.validateHaveValidPartitionIdAndName(thePartition);
        this.validatePartitionNameDoesntAlreadyExist(thePartition.getName());
        this.validIdUponCreation(thePartition);
        ourLog.info("Creating new partition with ID {} and Name {}", (Object)thePartition.getId(), (Object)thePartition.getName());
        PartitionEntity retVal = (PartitionEntity)this.myPartitionDao.save(thePartition);
        if (this.myInterceptorService.hasHooks((IPointcut)Pointcut.STORAGE_PARTITION_CREATED)) {
            HookParams params = new HookParams().add(RequestPartitionId.class, (Object)thePartition.toRequestPartitionId()).add(RequestDetails.class, (Object)theRequestDetails).addIfMatchesType(ServletRequestDetails.class, (Object)theRequestDetails);
            this.myInterceptorService.callHooks((IPointcut)Pointcut.STORAGE_PARTITION_CREATED, params);
        }
        return retVal;
    }

    @Override
    @Transactional
    public PartitionEntity updatePartition(PartitionEntity thePartition) {
        this.validateNotInUnnamedPartitionMode();
        this.validateHaveValidPartitionIdAndName(thePartition);
        Optional existingPartitionOpt = this.myPartitionDao.findById(thePartition.getId());
        if (!existingPartitionOpt.isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "unknownPartitionId", new Object[]{thePartition.getId()});
            throw new InvalidRequestException(Msg.code((int)1307) + msg);
        }
        PartitionEntity existingPartition = (PartitionEntity)existingPartitionOpt.get();
        if (!thePartition.getName().equalsIgnoreCase(existingPartition.getName())) {
            this.validatePartitionNameDoesntAlreadyExist(thePartition.getName());
        }
        existingPartition.setName(thePartition.getName());
        existingPartition.setDescription(thePartition.getDescription());
        this.myPartitionDao.save(existingPartition);
        this.invalidateCaches();
        return existingPartition;
    }

    @Override
    @Transactional
    public void deletePartition(Integer thePartitionId) {
        PartitionLookupSvcImpl.validatePartitionIdSupplied(this.myFhirCtx, thePartitionId);
        this.validateNotInUnnamedPartitionMode();
        Optional partition = this.myPartitionDao.findById(thePartitionId);
        if (!partition.isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "unknownPartitionId", new Object[]{thePartitionId});
            throw new IllegalArgumentException(Msg.code((int)1308) + msg);
        }
        this.myPartitionDao.delete((PartitionEntity)partition.get());
        if (this.myInterceptorService.hasHooks((IPointcut)Pointcut.STORAGE_PARTITION_DELETED)) {
            HookParams params = new HookParams().add(RequestPartitionId.class, (Object)((PartitionEntity)partition.get()).toRequestPartitionId());
            this.myInterceptorService.callHooks((IPointcut)Pointcut.STORAGE_PARTITION_DELETED, params);
        }
        this.invalidateCaches();
    }

    @Override
    public List<PartitionEntity> listPartitions() {
        List allPartitions = this.myPartitionDao.findAll();
        return allPartitions;
    }

    private void validatePartitionNameDoesntAlreadyExist(String theName) {
        if (this.myPartitionDao.findForName(theName).isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "cantCreateDuplicatePartitionName", new Object[]{theName});
            throw new InvalidRequestException(Msg.code((int)1309) + msg);
        }
    }

    private void validIdUponCreation(PartitionEntity thePartition) {
        if (this.myPartitionDao.findById(thePartition.getId()).isPresent()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "duplicatePartitionId", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)2366) + msg);
        }
    }

    private void validateHaveValidPartitionIdAndName(PartitionEntity thePartition) {
        if (thePartition.getId() == null || StringUtils.isBlank((CharSequence)thePartition.getName())) {
            String msg = this.myFhirCtx.getLocalizer().getMessage(PartitionLookupSvcImpl.class, "missingPartitionIdOrName", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)1310) + msg);
        }
        if (thePartition.getName().equals("DEFAULT")) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "cantCreateDefaultPartition", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)1311) + msg);
        }
        if (!PARTITION_NAME_VALID_PATTERN.matcher(thePartition.getName()).matches()) {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "invalidName", new Object[]{thePartition.getName()});
            throw new InvalidRequestException(Msg.code((int)1312) + msg);
        }
    }

    private void validateNotInUnnamedPartitionMode() {
        if (this.myPartitionSettings.isUnnamedPartitionMode()) {
            throw new MethodNotAllowedException(Msg.code((int)1313) + "Can not invoke this operation in unnamed partition mode");
        }
    }

    private PartitionEntity lookupPartitionByName(@Nonnull String theName) {
        return (PartitionEntity)((Optional)this.executeInTransaction(() -> this.myPartitionDao.findForName(theName))).orElseThrow(() -> {
            String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "invalidName", new Object[]{theName});
            return new ResourceNotFoundException(msg);
        });
    }

    private PartitionEntity lookupPartitionById(@Nonnull Integer theId) {
        try {
            return (PartitionEntity)((Optional)this.executeInTransaction(() -> this.myPartitionDao.findById(theId))).orElseThrow(() -> {
                String msg = this.myFhirCtx.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "unknownPartitionId", new Object[]{theId});
                return new ResourceNotFoundException(msg);
            });
        }
        catch (ResourceNotFoundException e) {
            List allPartitions = (List)this.executeInTransaction(() -> this.myPartitionDao.findAll());
            String allPartitionsString = allPartitions.stream().map(t -> t.getId() + "/" + t.getName()).collect(Collectors.joining(", "));
            ourLog.warn("Failed to find partition with ID {}.  Current partitions: {}", (Object)theId, (Object)allPartitionsString);
            throw e;
        }
    }

    protected <T> T executeInTransaction(ICallable<T> theCallable) {
        return (T)this.myTxTemplate.execute(tx -> theCallable.call());
    }

    public static void validatePartitionIdSupplied(FhirContext theFhirContext, Integer thePartitionId) {
        if (thePartitionId == null) {
            String msg = theFhirContext.getLocalizer().getMessageSanitized(PartitionLookupSvcImpl.class, "noIdSupplied", new Object[0]);
            throw new InvalidRequestException(Msg.code((int)1314) + msg);
        }
    }
}

