/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.fhir.batch2.maintenance;

import ca.uhn.fhir.batch2.api.IJobPersistence;
import ca.uhn.fhir.batch2.channel.BatchJobSender;
import ca.uhn.fhir.batch2.coordinator.WorkChunkProcessor;
import ca.uhn.fhir.batch2.maintenance.JobChunkProgressAccumulator;
import ca.uhn.fhir.batch2.model.JobInstance;
import ca.uhn.fhir.batch2.model.JobWorkCursor;
import ca.uhn.fhir.batch2.model.JobWorkNotification;
import ca.uhn.fhir.batch2.model.StatusEnum;
import ca.uhn.fhir.batch2.progress.JobInstanceProgressCalculator;
import ca.uhn.fhir.batch2.progress.JobInstanceStatusUpdater;
import ca.uhn.fhir.util.Logs;
import java.util.List;
import org.slf4j.Logger;

public class JobInstanceProcessor {
    private static final Logger ourLog = Logs.getBatchTroubleshootingLog();
    public static final long PURGE_THRESHOLD = 604800000L;
    private final IJobPersistence myJobPersistence;
    private final BatchJobSender myBatchJobSender;
    private final JobInstance myInstance;
    private final JobChunkProgressAccumulator myProgressAccumulator;
    private final JobInstanceProgressCalculator myJobInstanceProgressCalculator;
    private final WorkChunkProcessor myJobExecutorSvc;
    private final JobInstanceStatusUpdater myJobInstanceStatusUpdater;

    JobInstanceProcessor(IJobPersistence theJobPersistence, BatchJobSender theBatchJobSender, JobInstance theInstance, JobChunkProgressAccumulator theProgressAccumulator, WorkChunkProcessor theExecutorSvc) {
        this.myJobPersistence = theJobPersistence;
        this.myBatchJobSender = theBatchJobSender;
        this.myInstance = theInstance;
        this.myJobExecutorSvc = theExecutorSvc;
        this.myProgressAccumulator = theProgressAccumulator;
        this.myJobInstanceProgressCalculator = new JobInstanceProgressCalculator(theJobPersistence, theInstance, theProgressAccumulator);
        this.myJobInstanceStatusUpdater = new JobInstanceStatusUpdater(theJobPersistence);
    }

    public void process() {
        this.handleCancellation();
        this.cleanupInstance();
        this.triggerGatedExecutions();
    }

    private void handleCancellation() {
        if (this.myInstance.isPendingCancellationRequest()) {
            this.myInstance.setErrorMessage(this.buildCancelledMessage());
            this.myJobInstanceStatusUpdater.setCancelled(this.myInstance);
        }
    }

    private String buildCancelledMessage() {
        Object msg = "Job instance cancelled";
        if (this.myInstance.hasGatedStep()) {
            msg = (String)msg + " while running step " + this.myInstance.getCurrentGatedStepId();
        }
        return msg;
    }

    private void cleanupInstance() {
        switch (this.myInstance.getStatus()) {
            case QUEUED: {
                break;
            }
            case IN_PROGRESS: 
            case ERRORED: 
            case FINALIZE: {
                this.myJobInstanceProgressCalculator.calculateAndStoreInstanceProgress();
                break;
            }
            case COMPLETED: 
            case FAILED: 
            case CANCELLED: {
                if (!this.purgeExpiredInstance()) break;
                return;
            }
        }
        if (this.myInstance.isFinished() && !this.myInstance.isWorkChunksPurged()) {
            this.myInstance.setWorkChunksPurged(true);
            this.myJobPersistence.deleteChunks(this.myInstance.getInstanceId());
            this.myJobPersistence.updateInstance(this.myInstance);
        }
    }

    private boolean purgeExpiredInstance() {
        if (this.myInstance.getEndTime() != null) {
            long cutoff = System.currentTimeMillis() - 604800000L;
            if (this.myInstance.getEndTime().getTime() < cutoff) {
                ourLog.info("Deleting old job instance {}", (Object)this.myInstance.getInstanceId());
                this.myJobPersistence.deleteInstanceAndChunks(this.myInstance.getInstanceId());
                return true;
            }
        }
        return false;
    }

    private void triggerGatedExecutions() {
        String currentStepId;
        if (!this.myInstance.isRunning()) {
            ourLog.debug("JobInstance {} is not in a \"running\" state. Status {}", (Object)this.myInstance.getInstanceId(), (Object)this.myInstance.getStatus().name());
            return;
        }
        if (!this.myInstance.hasGatedStep()) {
            return;
        }
        JobWorkCursor<?, ?, ?> jobWorkCursor = JobWorkCursor.fromJobDefinitionAndRequestedStepId(this.myInstance.getJobDefinition(), this.myInstance.getCurrentGatedStepId());
        if (jobWorkCursor.isFinalStep() && !jobWorkCursor.isReductionStep()) {
            ourLog.debug("Job instance {} is in final step and it's not a reducer step", (Object)this.myInstance.getInstanceId());
            return;
        }
        if (jobWorkCursor.isReductionStep() && this.myInstance.getStatus() == StatusEnum.FINALIZE) {
            ourLog.warn("Job instance {} is still finalizing - a second reduction job will not be started.", (Object)this.myInstance.getInstanceId());
            return;
        }
        String instanceId = this.myInstance.getInstanceId();
        boolean shouldAdvance = this.myJobPersistence.canAdvanceInstanceToNextStep(instanceId, currentStepId = jobWorkCursor.getCurrentStepId());
        if (shouldAdvance) {
            String nextStepId = jobWorkCursor.nextStep.getStepId();
            ourLog.info("All processing is complete for gated execution of instance {} step {}. Proceeding to step {}", new Object[]{instanceId, currentStepId, nextStepId});
            if (jobWorkCursor.nextStep.isReductionStep()) {
                this.processReductionStep(jobWorkCursor);
            } else {
                this.processChunksForNextSteps(instanceId, nextStepId);
            }
        } else {
            ourLog.debug("Not ready to advance gated execution of instance {} from step {} to {}.", new Object[]{instanceId, currentStepId, jobWorkCursor.nextStep.getStepId()});
        }
    }

    private void processChunksForNextSteps(String instanceId, String nextStepId) {
        List<String> queuedChunksForNextStep = this.myProgressAccumulator.getChunkIdsWithStatus(instanceId, nextStepId, StatusEnum.QUEUED);
        int totalChunksForNextStep = this.myProgressAccumulator.getTotalChunkCountForInstanceAndStep(instanceId, nextStepId);
        if (totalChunksForNextStep != queuedChunksForNextStep.size()) {
            ourLog.debug("Total ProgressAccumulator QUEUED chunk count does not match QUEUED chunk size! [instanceId={}, stepId={}, totalChunks={}, queuedChunks={}]", new Object[]{instanceId, nextStepId, totalChunksForNextStep, queuedChunksForNextStep.size()});
        }
        List<String> chunksToSubmit = this.myJobPersistence.fetchallchunkidsforstepWithStatus(instanceId, nextStepId, StatusEnum.QUEUED);
        for (String nextChunkId : chunksToSubmit) {
            JobWorkNotification workNotification = new JobWorkNotification(this.myInstance, nextStepId, nextChunkId);
            this.myBatchJobSender.sendWorkChannelMessage(workNotification);
        }
        ourLog.debug("Submitted a batch of chunks for processing. [chunkCount={}, instanceId={}, stepId={}]", new Object[]{chunksToSubmit.size(), instanceId, nextStepId});
        this.myInstance.setCurrentGatedStepId(nextStepId);
        this.myJobPersistence.updateInstance(this.myInstance);
    }

    private void processReductionStep(JobWorkCursor<?, ?, ?> theWorkCursor) {
        JobWorkNotification workNotification = new JobWorkNotification(this.myInstance, theWorkCursor.nextStep.getStepId(), "REDUCTION");
        ourLog.debug("Submitting a Work Notification for a job reduction step. No associated work chunk ids are available.");
        this.myBatchJobSender.sendWorkChannelMessage(workNotification);
    }
}

