package cc.zhaoac.faith.tool.geotool.utils;

import cc.zhaoac.faith.tool.geotool.enums.GeoJsonType;
import cc.zhaoac.faith.tool.geotool.exception.GeoException;
import cc.zhaoac.faith.tool.geotool.exception.NoSupportTypeException;
import cc.zhaoac.faith.tool.geotool.pool.GeoJsonPool;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.file.FileNameUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONException;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.io.StringWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.geotools.data.DataUtilities;
import org.geotools.data.DefaultTransaction;
import org.geotools.data.FileDataStore;
import org.geotools.data.FileDataStoreFinder;
import org.geotools.data.Transaction;
import org.geotools.data.collection.ListFeatureCollection;
import org.geotools.data.geojson.GeoJSONReader;
import org.geotools.data.geojson.GeoJSONReader.IdStrategy;
import org.geotools.data.geojson.GeoJSONWriter;
import org.geotools.data.shapefile.ShapefileDataStore;
import org.geotools.data.shapefile.ShapefileDataStoreFactory;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.geotools.data.simple.SimpleFeatureSource;
import org.geotools.data.simple.SimpleFeatureStore;
import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.feature.NameImpl;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeImpl;
import org.geotools.feature.type.GeometryDescriptorImpl;
import org.geotools.feature.type.GeometryTypeImpl;
import org.geotools.kml.v22.KMLConfiguration;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.geotools.xsd.PullParser;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;
import org.locationtech.jts.io.WKTWriter;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.AttributeType;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.feature.type.GeometryType;

/**
 * 判断拓展名
 *
 * @author zhaoanchi
 * @version 1.1.0
 * @since 2023-10-18 16:27
 */
@Slf4j
@UtilityClass
public class GeoUtil {

    private static final String GEO_JSON = "geojson";

    /**
     * WKTReader
     */
    private static final WKTReader READER = new WKTReader();

    /**
     * 要素集合根节点
     */
    private static final String[] COLLECTION_TYPE = new String[]{"FeatureCollection"};

    /**
     * 地理要素类型
     */
    private static final String[] FEATURES_TYPE = new String[]{"Feature"};

    /**
     * 地理数据类型 点、线、面、几何集合
     */
    private static final String[] GEO_TYPE = new String[]{"Geometry", "Point", "LineString", "Polygon", "MultiPoint",
            "MultiLineString", "MultiPolygon", "GeometryCollection"};


    /**
     * 是规范的 WKT
     *
     * @param wktStr WKT 字符串
     * @return 是、否
     */
    public static boolean isWkt(String wktStr) {
        for (String s : GEO_TYPE) {
            if (wktStr.toLowerCase().startsWith(s.toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    /**
     * 不是规范的 WKT
     *
     * @param wktStr WKT 字符串
     * @return 是、否
     */
    public static boolean isNotWkt(String wktStr) {
        return !isWkt(wktStr);
    }

    /**
     * 是规范的 GeoJson
     *
     * @param geoJsonStr GeoJson 字符串
     * @return 是、否
     */
    public static boolean isGeoJson(String geoJsonStr) {
        try {
            JSONObject jsonObject = new JSONObject(geoJsonStr);
            return isGeoJson(jsonObject);
        } catch (Exception e) {
            log.warn(e.getMessage());
            return false;
        }
    }

    /**
     * 是规范的 GeoJson
     *
     * @param geoJson GeoJson 对象
     * @return 是、否
     */
    public static boolean isGeoJson(JSONObject geoJson) {
        try {
            String type = geoJson.getStr(GeoJsonPool.TYPE);
            boolean mark = false;
            // 判断根节点
            if (ArrayUtil.containsIgnoreCase(COLLECTION_TYPE, type)) {
                JSONArray jsonArray = geoJson.get(GeoJsonPool.FEATURES, JSONArray.class);
                Iterator<Object> iterator = jsonArray.iterator();
                while (iterator.hasNext()) {
                    Object jsonStr = iterator.next();
                    JSONObject jsonObject = JSONUtil.parseObj(jsonStr);
                    type = jsonObject.getStr(GeoJsonPool.TYPE);
                    // 判断地理要素
                    if (ArrayUtil.containsIgnoreCase(FEATURES_TYPE, type)) {
                        JSONObject geometry = jsonObject.get(GeoJsonPool.GEOMETRY, JSONObject.class);
                        if (geometry == null) {
                            iterator.remove();
                            continue;
                        }
                        type = geometry.getStr(GeoJsonPool.TYPE);
                        // 判断几何要素
                        mark = ArrayUtil.containsIgnoreCase(GEO_TYPE, type);
                    }
                    if (!mark) {
                        return false;
                    }
                }
            } else {
                // 判断地理要素
                if (ArrayUtil.containsIgnoreCase(FEATURES_TYPE, type)) {
                    type = geoJson.get(GeoJsonPool.GEOMETRY, JSONObject.class).getStr(GeoJsonPool.TYPE);
                }
                // 数据是几何数据
                mark = ArrayUtil.containsIgnoreCase(GEO_TYPE, type);
            }
            return mark;
        } catch (JSONException e) {
            throw new JSONException(e);
        }
    }

    /**
     * 不是规范的 GeoJson
     *
     * @param geoJson GeoJson 对象
     * @return 是、否
     */
    public static boolean isNotGeoJson(JSONObject geoJson) {
        return !isGeoJson(geoJson);
    }

    /**
     * 获取 Geo 几何类型
     *
     * @param wktStr WKT 字符串
     * @return Geo 几何类型
     */
    public static String getGeometryType(String wktStr) {
        String type = null;
        if (CharSequenceUtil.isNotEmpty(wktStr)) {
            try {
                Geometry read = READER.read(wktStr);
                type = read.getGeometryType();
            } catch (Exception e) {
                log.warn("非规范 WKT 字符串：{}", e.getMessage());
            }
        }
        return type;
    }

    /**
     * wkt 转 Geometry
     *
     * @param geometry geometry
     * @return wkt
     */
    public String geometryToWkt(Geometry geometry) {
        WKTWriter writer = new WKTWriter();
        return writer.write(geometry);
    }

    /**
     * geometry 转 geoJson
     *
     * @param geometry geometry
     * @return GeometryJSON
     */
    public String geometryToGeometryJson(Geometry geometry) {
        String wkt = geometryToWkt(geometry);
        return wktToGeoJson(wkt);
    }

    /**
     * GeometryJSON 转 Geometry
     *
     * @param geoJson geoJson
     * @return Geometry
     */
    public Geometry geometryJsonToGeometry(String geoJson) {
        String wkt = geoJsonToWkt(geoJson);
        return wktToGeometry(wkt);
    }

    /**
     * wkt 转 Geometry
     *
     * @param wkt wkt
     * @return Geometry
     */
    public Geometry wktToGeometry(String wkt) {
        WKTReader reader = new WKTReader();
        try {
            return reader.read(wkt);
        } catch (ParseException e) {
            throw new GeoException(e);
        }
    }

    /**
     * GeoJson 转 WKT
     *
     * @param geoJson GeoJson 对象
     * @return WKT 字符串
     */
    public static String geoJsonToWkt(String geoJson) {
        if (CharSequenceUtil.isBlank(geoJson)) {
            return null;
        }

        String type = new JSONObject(geoJson).getStr(GeoJsonPool.TYPE);
        if (CharSequenceUtil.equalsIgnoreCase(type, GeoJsonType.FEATURE_COLLECTION.getName())) {
            SimpleFeatureCollection simpleFeatureCollection = geoJsonToFeatureCollection(geoJson);
            Geometry geometry = featureCollectionToGeometry(simpleFeatureCollection);
            return geometry.toText();
        } else if (CharSequenceUtil.equalsIgnoreCase(type, GeoJsonType.FEATURE.getName())) {
            return Optional.ofNullable(geoJsonToFeature(geoJson)).map(GeoUtil::featureToGeometry).map(Geometry::toText)
                    .orElse(null);
        } else {
            throw new NoSupportTypeException("only support FeatureCollection and Feature");
        }
    }

    /**
     * WKT 转 GeoJson
     *
     * @param wktStr WKT 字符串
     * @return GeoJson字符串
     */
    public static String wktToGeoJson(String wktStr) {
        if (CharSequenceUtil.isBlank(wktStr)) {
            return null;
        }

        SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
        builder.setCRS(DefaultGeographicCRS.WGS84);
        builder.setName(GEO_JSON);
        builder.add(GeoJsonPool.GEOMETRY, Geometry.class);

        int idVal = 0;
        DefaultFeatureCollection featureCollection = new DefaultFeatureCollection();
        // wkt写入FeatureCollection
        SimpleFeatureType featureType = builder.buildFeatureType();
        SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(featureType);
        Geometry geometry = GeoUtil.wktToGeometry(wktStr);
        if (geometry instanceof GeometryCollection) {
            GeometryCollection geometryCollection = (GeometryCollection) geometry;
            for (int i = 0; i < geometryCollection.getNumGeometries(); i++) {
                featureBuilder.add(geometryCollection.getGeometryN(i));
                SimpleFeature feature = featureBuilder.buildFeature(String.valueOf(idVal++));
                featureCollection.add(feature);
            }
        } else {
            featureBuilder.add(geometry);
            SimpleFeature feature = featureBuilder.buildFeature(String.valueOf(idVal));
            featureCollection.add(feature);
            // 输出geojson
        }
        return GeoJSONWriter.toGeoJSON(featureCollection);
    }

    /**
     * Feature 转 Geometry
     *
     * @param feature feature
     * @return Geometry
     */
    public static Geometry featureToGeometry(SimpleFeature feature) {
        return (Geometry) feature.getDefaultGeometry();

    }

    /**
     * FeatureCollection 转 Geometry
     *
     * @param featureCollection featureCollection
     * @return GeometryCollection
     */
    public static Geometry featureCollectionToGeometry(SimpleFeatureCollection featureCollection) {
        SimpleFeatureIterator features = featureCollection.features();
        Geometry[] geometries = new Geometry[featureCollection.size()];
        int seek = 0;
        while (features.hasNext()) {
            SimpleFeature feature = features.next();
            geometries[seek++] = featureToGeometry(feature);
        }
        if (geometries.length == 1) {
            return geometries[0];
        } else {
            GeometryFactory geometryFactory = new GeometryFactory(new PrecisionModel(), 4326);
            return new GeometryCollection(geometries, geometryFactory);
        }
    }

    /**
     * geoJson 转 Feature
     *
     * @param geoJson geoJson
     * @return Feature
     */
    public static SimpleFeature geoJsonToFeature(String geoJson) {
        try {
            return GeoJSONReader.parseFeature(geoJson);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * geoJson 转 Feature
     *
     * @param geoJson     geoJson
     * @param idStrategy  Auto generated, prefixed or provided id
     * @param idPrefix    Prefix for id
     * @param idFieldName Field name for existing id in json
     * @return Feature
     */
    public static SimpleFeature geoJsonToFeature(String geoJson, IdStrategy idStrategy, String idPrefix,
            String idFieldName) {
        try {
            return GeoJSONReader.parseFeature(geoJson, idStrategy, idPrefix, idFieldName);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * geoJson 转 FeatureCollection
     *
     * @param geoJson geoJson
     * @return FeatureCollection
     */
    public SimpleFeatureCollection geoJsonToFeatureCollection(String geoJson) {
        String type = new JSONObject(geoJson).getStr(GeoJsonPool.TYPE);
        if (CharSequenceUtil.equalsIgnoreCase(type, GeoJsonType.FEATURE_COLLECTION.getName())) {
            return GeoJSONReader.parseFeatureCollection(geoJson);
        } else if (CharSequenceUtil.equalsIgnoreCase(type, GeoJsonType.FEATURE.getName())) {
            SimpleFeature feature = geoJsonToFeature(geoJson);
            DefaultFeatureCollection simpleFeatureCollection = new DefaultFeatureCollection();
            simpleFeatureCollection.add(feature);
            return simpleFeatureCollection;
        } else {
            throw new NoSupportTypeException("only support FeatureCollection and Feature");
        }
    }

    /**
     * geojson(标准格式) 转 shp
     *
     * @param geojson geojson
     * @param shpPath shp路径, 例:/data/shp/demo.shp
     * @return Boolean false=异常 true成功
     */
    public static Boolean geoJson2Shp(String geojson, String shpPath) {
        SimpleFeatureCollection simpleFeatureCollection = GeoUtil.geoJsonToFeatureCollection(geojson);
        return generateShp(simpleFeatureCollection, shpPath);
    }

    /**
     * 读取shp文件
     *
     * @param shpPath shp路径, 例:/data/shp/demo/shp
     * @return true=成功 false=异常
     */
    public static boolean wktToShp(String wkt, String shpPath) {
        String geoJson = wktToGeoJson(wkt);
        return geoJson2Shp(geoJson, shpPath);
    }

    /**
     * 读取shp文件
     *
     * @param file file
     * @return List&#60;Map&#60;String, Object&#62;&#62;
     */
    public static List<Map<String, Object>> shpToWkt(File file) throws IOException {
        //加载文件
        //map记录shapefile key-value数据
        List<Map<String, Object>> list = new ArrayList<>();
        //通过store获取featurecollection
        FileDataStore store = FileDataStoreFinder.getDataStore(file);
        SimpleFeatureSource featureSource = store.getFeatureSource();
        SimpleFeatureCollection simpleFeatureCollection = featureSource.getFeatures();
        SimpleFeatureIterator itertor = simpleFeatureCollection.features();
        //遍历featurecollection
        while (itertor.hasNext()) {
            Map<String, Object> data = new HashMap<>(16);
            SimpleFeature feature = itertor.next();
            Collection<Property> p = feature.getProperties();
            //遍历feature的properties
            for (Property pro : p) {
                String field = pro.getName().toString();
                String value = pro.getValue().toString();
                field = "the_geom".equals(field) ? "wkt" : field;
                byte[] bytes = value.getBytes();
                value = new String(bytes);
                data.put(field, value);
            }
            list.add(data);
        }
        itertor.close();
        return list;
    }

    /**
     * kml转换为geojson
     *
     * @param input 输入
     * @return {@link StringWriter}
     * @throws IOException ioexception
     */
    public static String kmlToGeoJson(InputStream input) throws IOException {
        try {
            DefaultFeatureCollection featureCollection = new DefaultFeatureCollection();
            PullParser parser = new PullParser(new KMLConfiguration(), input, SimpleFeature.class);
            SimpleFeature simpleFeature = (SimpleFeature) parser.parse();
            while (simpleFeature != null) {
                if (simpleFeature.getDefaultGeometry() != null) {
                    featureCollection.add(simpleFeature);
                }
                simpleFeature = (SimpleFeature) parser.parse();
            }
            return GeoJSONWriter.toGeoJSON(featureCollection);
        } catch (Exception e) {
            log.error("KML 文件解析报错：{}", e.getMessage());
            return null;
        } finally {
            Objects.requireNonNull(input, "KML 文件输入流为空").close();
        }
    }


    /* ****************************** private ****************************** */
    /* ****************************** private ****************************** */

    /**
     * 设置字符编码 防止乱码 创建cpg文件对象
     */
    private static void generateCpgFile(String localDirPath, String name, Charset charset) {
        File file = new File(localDirPath + File.separator + name + ".cpg");
        try (BufferedWriter writer = new BufferedWriter(new FileWriter(file))) {
            writer.write(charset.toString());
            writer.flush();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 根据FeatureCollection生成shp文件
     *
     * @param features features
     * @param shpPath  shp路径, 例:/data/shp/demo.shp
     * @return true=成功 false=失败
     */
    private boolean generateShp(SimpleFeatureCollection features, String shpPath) {
        // convert schema for shapefile
        SimpleFeatureType schema = features.getSchema();
        GeometryDescriptor geom = schema.getGeometryDescriptor();
        // geojson文件属性
        List<AttributeDescriptor> attributes = schema.getAttributeDescriptors();
        // geojson文件空间类型（必须在第一个）
        GeometryType geomType = null;
        List<AttributeDescriptor> attribs = new ArrayList<>();
        for (AttributeDescriptor attrib : attributes) {
            AttributeType type = attrib.getType();
            if (type instanceof GeometryType) {
                geomType = (GeometryType) type;
            } else {
                attribs.add(attrib);
            }
        }

        if (geomType == null) {
            return false;
        }

        // 使用geomType创建gt
        GeometryTypeImpl gt = new GeometryTypeImpl(new NameImpl("the_geom"), geomType.getBinding(),
                geom.getCoordinateReferenceSystem() == null ? DefaultGeographicCRS.WGS84
                        : geom.getCoordinateReferenceSystem(), // 用户未指定则默认为wgs84
                geomType.isIdentified(), geomType.isAbstract(), geomType.getRestrictions(), geomType.getSuper(),
                geomType.getDescription());

        // 创建识别符
        GeometryDescriptor geomDesc = new GeometryDescriptorImpl(gt, new NameImpl("the_geom"), geom.getMinOccurs(),
                geom.getMaxOccurs(), geom.isNillable(), geom.getDefaultValue());

        // the_geom 属性必须在第一个
        attribs.add(0, geomDesc);

        SimpleFeatureType outSchema = new SimpleFeatureTypeImpl(schema.getName(), attribs, geomDesc,
                schema.isAbstract(), schema.getRestrictions(), schema.getSuper(), schema.getDescription());
        List<SimpleFeature> outFeatureList = new ArrayList<>();
        try (FeatureIterator<SimpleFeature> featuresIterator = features.features()) {
            while (featuresIterator.hasNext()) {
                SimpleFeature f = featuresIterator.next();
                SimpleFeature reType = DataUtilities.reType(outSchema, f, true);

                reType.setAttribute(outSchema.getGeometryDescriptor().getName(),
                        f.getAttribute(schema.getGeometryDescriptor().getName()));

                outFeatureList.add(reType);
            }
        }
        return saveFeaturesToShp(outFeatureList, outSchema, shpPath);
    }

    /**
     * 保存features为shp格式
     *
     * @param features 要素类
     * @param type     要素类型
     * @param shpPath  shp路径, 例:/data/shp/demo/shp
     * @return 是否保存成功
     */
    private static boolean saveFeaturesToShp(List<SimpleFeature> features, SimpleFeatureType type, String shpPath) {
        try {
            ShapefileDataStoreFactory dataStoreFactory = new ShapefileDataStoreFactory();
            File shpFile = new File(shpPath);
            String parentPath = shpFile.getParent();
            FileUtil.mkdir(parentPath);
            String name = FileNameUtil.getPrefix(shpPath);
            Map<String, Serializable> params = new HashMap<>(5);
            params.put("url", shpFile.toURI().toURL());
            params.put("create spatial index", Boolean.TRUE);

            ShapefileDataStore newDataStore = (ShapefileDataStore) dataStoreFactory.createNewDataStore(params);
            newDataStore.setCharset(StandardCharsets.UTF_8);

            newDataStore.createSchema(type);

            Transaction transaction = new DefaultTransaction("create");
            String typeName = newDataStore.getTypeNames()[0];
            SimpleFeatureSource featureSource = newDataStore.getFeatureSource(typeName);

            if (featureSource instanceof SimpleFeatureStore) {
                SimpleFeatureStore featureStore = (SimpleFeatureStore) featureSource;
                SimpleFeatureCollection collection = new ListFeatureCollection(type, features);
                featureStore.setTransaction(transaction);
                try {
                    featureStore.addFeatures(collection);
                    generateCpgFile(parentPath, name, StandardCharsets.UTF_8);
                    transaction.commit();
                } catch (Exception problem) {
                    problem.printStackTrace();
                    transaction.rollback();
                } finally {
                    transaction.close();
                }
            } else {
                log.warn(typeName + " does not support read/write access");
            }
            return true;
        } catch (IOException e) {
            log.warn("保存shp失败, {}", e.getMessage());
            return false;
        }
    }
}
