/*
 * Id$: zuv-cloud:z-job-support:cc.zuv.job.support.impl.quartz.QuartzService:20190104164249
 *
 * QuartzService.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.job.support.impl.quartz;

import cc.zuv.ZuvException;
import cc.zuv.job.support.IJobCode;
import cc.zuv.job.support.core.IScheduleService;
import cc.zuv.job.support.data.domain.JobsFireEntity;
import cc.zuv.job.support.data.domain.JobsTaskEntity;
import cc.zuv.job.support.impl.quartz.demo.DemoNativeExecutor;
import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.quartz.impl.matchers.EverythingMatcher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-12-24 20:21
 * @modify.date     2018-12-24 20:21
 */
@Slf4j
@Service
public class QuartzService implements IScheduleService, IJobCode
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    private Scheduler scheduler;

    //-----------------------------------------------------------------------------------------

    public String identityJobKey(JobKey jobKey)
    {
        return jobKey.getName() + " @ " + jobKey.getGroup();
    }

    public String identityTriggerKey(TriggerKey triggerKey)
    {
        return triggerKey.getName() + " @ " + triggerKey.getGroup();
    }

    private JobKey bldJobKey(JobsTaskEntity task)
    {
        return JobKey.jobKey(task.getName(), task.getTeam());
    }
    private JobKey bldJobKey(JobsFireEntity fire)
    {
        return JobKey.jobKey(fire.getName(), fire.getTeam());
    }

    private JobDataMap bldJobDataMap(JobsTaskEntity task)
    {
        JobDataMap jobDataMap = new JobDataMap();
        jobDataMap.put(JOBS_MAPDATA_NAME_TASKID, task.getId());
//        jobDataMap.put(JOBS_MAPDATA_NAME_TASKDATA, task.getData());
//        jobDataMap.put(JOBS_MAPDATA_NAME_TASKRECORD, task.getRecord());
        return jobDataMap;
    }

    private JobDataMap bldJobDataMap(JobsFireEntity fire)
    {
        JobDataMap jobDataMap = new JobDataMap();
        jobDataMap.put(JOBS_MAPDATA_NAME_FIREID, fire.getId());
//        jobDataMap.put(JOBS_MAPDATA_NAME_FIREREQCODE, fire.getReqcode());
//        jobDataMap.put(JOBS_MAPDATA_NAME_FIREREQDATA, fire.getReqdata());
        return jobDataMap;
    }

    @SuppressWarnings(value = "unchecked")
    private JobDetail bldJobDetail(JobsTaskEntity task)
    {
        try
        {
            String execclassname = task.getClsname();
            Class<? extends Job> execClass = (Class<? extends Job>) Class.forName(execclassname).newInstance().getClass();

            JobKey jobKey = bldJobKey(task);
            JobDataMap jobDataMap = bldJobDataMap(task);
            String desc = task.getMemo();
            return bldJobDetail(execClass, jobKey, jobDataMap, desc);
        }
        catch (ClassNotFoundException | InstantiationException | IllegalAccessException e)
        {
            log.error("类{}实例化失败: {}", task.getClsname(), e.getMessage());
            throw new ZuvException(String.format("类%s实例化失败", task.getClsname()), e);
        }
        /*
        Class<? extends Job> execClass = task.getType().getCls();
        JobKey jobKey = bldJobKey(task);
        JobDataMap jobDataMap = bldJobDataMap(task);
        String desc = task.getMemo();
        return bldJobDetail(execClass, jobKey, jobDataMap, desc);
        */
    }

    @SuppressWarnings(value = "unchecked")
    private JobDetail bldJobDetail(JobsFireEntity fire)
    {
        try
        {
            String execclassname = fire.getClsname();
            Class<? extends Job> execClass = (Class<? extends Job>) Class.forName(execclassname).newInstance().getClass();

            JobKey jobKey = bldJobKey(fire);
            JobDataMap jobDataMap = bldJobDataMap(fire);
            String desc = fire.getMemo();
            return bldJobDetail(execClass, jobKey, jobDataMap, desc);
        }
        catch (ClassNotFoundException | InstantiationException | IllegalAccessException e)
        {
            log.error("类{}实例化失败: {}", fire.getClsname(), e.getMessage());
            throw new ZuvException(String.format("类%s实例化失败", fire.getClsname()), e);
        }
        /*
        Class<? extends Job> execClass = fire.getType().getCls();
        JobKey jobKey = bldJobKey(fire);
        JobDataMap jobDataMap = bldJobDataMap(fire);
        String desc = fire.getMemo();
        return bldJobDetail(execClass, jobKey, jobDataMap, desc);
        */
    }

    public JobDetail bldJobDetail(Class<? extends Job> execClass, JobKey jobKey, JobDataMap jobDataMap, String desc)
    {
        return JobBuilder.newJob(execClass)
            .withIdentity(jobKey)
            .withDescription(desc)
            .setJobData(jobDataMap)
            .storeDurably()
            .build();
    }

    //-----------------------------------------------------------------------------------------

    public void addJobListener(JobListener listener)
    {
        try
        {
            scheduler.getListenerManager().addJobListener(listener, EverythingMatcher.allJobs());
        }
        catch (SchedulerException e)
        {
            log.info("add listener {} failure: {}", listener.getName(), e.getMessage());
            throw new ZuvException(String.format("add listener %s failure", listener.getName()), e);
        }
    }

    //-----------------------------------------------------------------------------------------

    public JobDetail getJobDetail(JobKey jobKey)
    {
        try
        {
            return scheduler.getJobDetail(jobKey);
        }
        catch (SchedulerException e)
        {
            log.error("获取JobDetail失败: {}", e.getMessage());
            throw new ZuvException("获取JobDetail失败" ,e);
        }
    }

    public boolean containJobDetail(JobsTaskEntity task)
    {
        JobKey jobKey = bldJobKey(task);
        return getJobDetail(jobKey)!=null;
    }

    public boolean containJobDetail(JobsFireEntity fire)
    {
        JobKey jobKey = bldJobKey(fire);
        return getJobDetail(jobKey)!=null;
    }

    public Trigger getTrigger(TriggerKey triggerKey)
    {
        try
        {
            return scheduler.getTrigger(triggerKey);
        }
        catch (SchedulerException e)
        {
            log.error("获取Trigger失败: {}", e.getMessage());
            throw new ZuvException("获取Trigger失败" ,e);
        }
    }

    public boolean containTrigger(JobsTaskEntity task)
    {
        TriggerKey triggerKey = QuartzTrigger.bldTriggerKey(task);
        return getTrigger(triggerKey)!=null;
    }

    public boolean containTrigger(JobsFireEntity fire)
    {
        TriggerKey triggerKey = QuartzTrigger.bldTriggerKey(fire);
        return getTrigger(triggerKey)!=null;
    }

    //-----------------------------------------------------------------------------------------

    public static void validateCronExpression(JobsTaskEntity task)
    {
        String cron = task.getCron();
        if (!CronExpression.isValidExpression(cron))
        {
            String message = String.format("Task %s expression %s is not correct", task.getClsname(), cron);
            throw new ZuvException(message);
        }
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public void createTask(JobsTaskEntity task)
    {
        //
        validateCronExpression(task);

        //
        JobDetail jobDetail = bldJobDetail(task);
        if (!task.getRestart()) //不需要立即启动
        {
            createTask(jobDetail);
        }
        else
        {
            if(containTrigger(task))
            {
                startTask(task);
            }
            else
            {
                CronTrigger cronTrigger = QuartzTrigger.bldCronTrigger(task);
                createTask(jobDetail, cronTrigger);
            }
        }
    }

    @Override
    public void updateTask(JobsTaskEntity task)
    {
        //
        validateCronExpression(task);

        //
        TriggerKey triggerKey = QuartzTrigger.bldTriggerKey(task);
        CronTrigger cronTrigger = QuartzTrigger.bldCronTrigger(triggerKey, task.getCron(), task.getMemo());
        updateTask(cronTrigger);

        //
        if (!task.getRestart()) //不需要立即启动
        {
            JobKey jobKey = bldJobKey(task);
            pauseTask(jobKey);
        }
    }

    @Override
    public void deleteTask(JobsTaskEntity task)
    {
        validateCronExpression(task);
        JobKey jobKey = bldJobKey(task);
        deleteTask(jobKey);
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public void startTask(JobsTaskEntity task)
    {
        validateCronExpression(task);
        JobKey jobKey = bldJobKey(task);
        startTask(jobKey);
    }

    @Override
    public void pauseTask(JobsTaskEntity task)
    {
        validateCronExpression(task);
        JobKey jobKey = bldJobKey(task);
        pauseTask(jobKey);
    }

    @Override
    public void resumeTask(JobsTaskEntity task)
    {
        validateCronExpression(task);
        JobKey jobKey = bldJobKey(task);
        resumeTask(jobKey);
    }

    //-----------------------------------------------------------------------------------------

    @Override
    public void createTask(JobsFireEntity fire)
    {
        JobKey jobKey = bldJobKey(fire);
        JobDetail jobDetail = getJobDetail(jobKey);
        if(jobDetail!=null)
        {
            jobDetail.getJobDataMap().putAll(bldJobDataMap(fire));
            createTask(jobDetail);
            startTask(fire);
        }
        else
        {
            jobDetail = bldJobDetail(fire);
            SimpleTrigger simpleTrigger = QuartzTrigger.bldSecondTrigger(fire, 1, 0);
            createTask(jobDetail, simpleTrigger);
        }
    }

    @Override
    public void deleteTask(JobsFireEntity fire)
    {
        JobKey jobKey = bldJobKey(fire);
        deleteTask(jobKey);
    }

    @Override
    public void startTask(JobsFireEntity fire)
    {
        JobKey jobKey = bldJobKey(fire);
        startTask(jobKey);
    }

    @Override
    public void pauseTask(JobsFireEntity fire)
    {
        JobKey jobKey = bldJobKey(fire);
        pauseTask(jobKey);
    }

    @Override
    public void resumeTask(JobsFireEntity fire)
    {
        JobKey jobKey = bldJobKey(fire);
        resumeTask(jobKey);
    }

    //-----------------------------------------------------------------------------------------

    //创建,不立即执行
    public void createTask(JobDetail jobDetail)
    {
        String identity = identityJobKey(jobDetail.getKey());
        try
        {
            scheduler.addJob(jobDetail, true);
            log.info("create task jobkey({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("create task jobkey({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("create task jobkey(%s) failure", identity), e);
        }
    }

    //创建,立即执行
    public void createTask(JobDetail jobDetail, Trigger trigger)
    {
        String identity = identityJobKey(jobDetail.getKey());
        try
        {
            scheduler.scheduleJob(jobDetail, trigger);
            log.info("create task jobkey({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("create task jobkey({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("create task jobkey(%s) failure", identity), e);
        }
    }

    public void updateTask(Trigger trigger)
    {
        TriggerKey triggerKey = trigger.getKey();
        String identity = identityTriggerKey(triggerKey);

        try
        {
            scheduler.rescheduleJob(triggerKey, trigger);
            log.info("update task triggerkey ({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("update task triggerkey ({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("update task triggerkey(%s) failure", identity), e);
        }
    }

    public void deleteTask(JobKey jobKey)
    {
        String identity = identityJobKey(jobKey);
        try
        {
            scheduler.deleteJob(jobKey);
            log.info("delete task jobkey({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("delete task jobkey({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("delete task jobkey(%s) failure", identity), e);
        }
    }


    public void startTask(JobKey jobKey)
    {
        String identity = identityJobKey(jobKey);
        try
        {
            scheduler.triggerJob(jobKey);
            log.info("start task jobkey({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("start task jobkey({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("start task jobkey(%s) failure", identity), e);
        }
    }

    public void pauseTask(JobKey jobKey)
    {
        String identity = identityJobKey(jobKey);
        try
        {
            scheduler.pauseJob(jobKey);
            log.info("pause task jobkey({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("pause task jobkey({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("pause task jobkey(%s) failure", identity), e);
        }
    }

    public void resumeTask(JobKey jobKey)
    {
        String identity = identityJobKey(jobKey);
        try
        {
            scheduler.resumeJob(jobKey);
            log.info("resume task jobkey({}) success", identity);
        }
        catch (SchedulerException e)
        {
            log.info("resume task jobkey({}) failure: {}", identity, e.getMessage());
            throw new ZuvException(String.format("resume task jobkey(%s) failure", identity), e);
        }
    }

    //-----------------------------------------------------------------------------------------

    public Boolean native_test()
    {
        Trigger trigger = TriggerBuilder.newTrigger()
            .withIdentity("trigger-native-test", "trigger-group-default")
            .startNow()//一旦加入scheduler，立即生效
            .withSchedule(SimpleScheduleBuilder.simpleSchedule() //使用SimpleTrigger
                .withIntervalInSeconds(5) //每隔5秒执行一次
                .repeatForever()) //一直执行，奔腾到老不停歇
            .build();

        JobDetail jobDetail = JobBuilder.newJob(DemoNativeExecutor.class)
            .withIdentity("job-native-test", "group-default")
            .withDescription("原生测试任务")
            .usingJobData("data", "{ type: 1}")
            .build();

        try
        {
            scheduler.scheduleJob(jobDetail, trigger);
            return true;
        }
        catch (SchedulerException e)
        {
            log.info("native test failure: {}", e.getMessage());
            return false;
        }
    }

    //-----------------------------------------------------------------------------------------


}
