/*
 * Id$: zuv-cloud:z-job-support:cc.zuv.job.support.impl.quartz.executor.QuartzScanExecutor:20190104164249
 *
 * QuartzScanExecutor.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.job.support.impl.quartz.executor;

import cc.zuv.job.support.IJobCode;
import cc.zuv.job.support.core.IExecContext;
import cc.zuv.job.support.core.IExecutor;
import cc.zuv.job.support.data.dao.JobsTaskEntityRepository;
import cc.zuv.job.support.data.dao.JobsWorkEntityRepository;
import cc.zuv.job.support.data.domain.JobsTaskEntity;
import cc.zuv.job.support.data.domain.JobsWorkEntity;
import cc.zuv.job.support.data.enums.TaskStatus;
import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * 任务扫描操作,不允许并发
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2018-12-24 17:26
 * @modify.date     2018-12-24 17:26
 */
@Slf4j
@PersistJobDataAfterExecution
@DisallowConcurrentExecution
public abstract class QuartzScanExecutor implements IExecutor, IJobCode, Job
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    protected JobsTaskEntityRepository jobTaskEntityRepository;

    @Autowired
    protected JobsWorkEntityRepository jobsWorkEntityRepository;

    //-----------------------------------------------------------------------------------------

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException
    {
        //
        JobDetail detail = context.getJobDetail();
        if(detail.getJobDataMap()==null
            || !detail.getJobDataMap().containsKey(JOBS_MAPDATA_NAME_TASKID)) return;

        //
        String key = detail.getKey().getName() + " @ " + detail.getKey().getGroup();
        String desc = detail.getDescription();
        Long taskid = detail.getJobDataMap().getLong(JOBS_MAPDATA_NAME_TASKID);
        log.debug("{} ({}) : {}", key, desc, taskid);

        //
        JobsTaskEntity taskEntity = jobTaskEntityRepository.findOne(taskid);
        if(taskEntity==null)
        {
            log.error("JobsFireEntity(id={}) not found.", taskid);
            return;
        }
        //初始数据在Listener中已经处理

        try
        {
            execute(new QuartzExecContext(detail, taskEntity));
        }
        catch (Exception e)
        {
            log.error("execute error {}", e.getMessage());
            //错误数据在监听中已经处理

//            // 策略1: 修复错误并立即重新启动
//            JobExecutionException e2 = new JobExecutionException(e);
//            detail.getJobDataMap().put("fixed", true);
//            e2.setRefireImmediately(true);

            // 策略2: 自动取消与此作业相关联的所有触发器, 以使其不再运行
            JobExecutionException e2 = new JobExecutionException(e);
            e2.setUnscheduleFiringTrigger(true);
//            e2.setUnscheduleAllTriggers(true);
            throw e2;
        }
    }

    public abstract void execute(IExecContext context);


    //-----------------------------------------------------------------------------------------

    protected void saveTaskEntity(JobsTaskEntity entity, TaskStatus status,
        Long beginTime, Long endTime, Long beginid, Long endid, String snapshot, String message)
    {
        entity.setStatus(status);
        entity.setLastBeginTime(beginTime);
        entity.setLastEndTime(endTime);
        entity.setLastBeginId(beginid);
        entity.setLastEndId(endid);
        entity.setLastSnapshot(snapshot);
        entity.setLastMessage(message);
        jobTaskEntityRepository.save(entity);
    }

    protected void saveWorkEntity(TaskStatus status, long taskid, String taskdata,
        Long beginTime, Long endTime, Long beginid, Long endid, Long effectnum, String snapshot, String message)
    {
        JobsWorkEntity workEntity = new JobsWorkEntity();
        workEntity.setTaskid(taskid);
        workEntity.setData(taskdata);

        workEntity.setBeginTime(beginTime);
        workEntity.setEndTime(endTime);
        workEntity.setBeginId(beginid);
        workEntity.setEndId(endid);
        workEntity.setEffectNum(effectnum);

        workEntity.setSnapshot(snapshot);
        workEntity.setMessage(message);
        workEntity.setStatus(status);
        jobsWorkEntityRepository.save(workEntity);
    }

    //-----------------------------------------------------------------------------------------

    protected void begin()
    {
        log.info("\n");
        log.info("-----------------------------------");
        log.info("BEGIN");
    }

    protected void end()
    {
        log.info("END");
        log.info("-----------------------------------");
    }

    //-----------------------------------------------------------------------------------------

}
