/*
 * Id$: zuv-cloud:z-web-support:cc.zuv.web.support.jwt.JwtParser:20190102094418
 *
 * JwtParser.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.web.support.jwt;

import cc.zuv.lang.StringUtils;
import cc.zuv.web.support.exception.RestAuthorizeException;
import io.jsonwebtoken.*;

/**
 * File Description
 *
 * @author			Kama Luther
 * @version			0.1
 * @since           0.1
 * @create.date     2014-2-11 下午02:19:48
 * @modify.date     2014-2-11 下午02:19:48
 */
public class JwtParser
{

    //  {"typ":"JWT","alg":"HS512","zip":"DEF"}
    //


    /*
    Header//头信息
    {
        "alg": "HS256",//签名或摘要算法
        "typ": "JWT"//token类型
    }
    Playload//荷载信息
    {
        "iss": "token-server",//签发者
        "exp ": "Mon Nov 13 15:28:41 CST 2017",//过期时间
        "sub ": "luther",//用户名
        "aud": "web-server-1"//接收方,
        "nbf": "Mon Nov 13 15:40:12 CST 2017",//这个时间之前token不可用
        "jat": "Mon Nov 13 15:20:41 CST 2017",//签发时间
        "jti": "0023",//令牌id标识
        "claim": {"auth":"ROLE_ADMIN"}//访问主张
    }
    Signature//签名信息
    签名或摘要算法(
        base64urlencode(Header),
        Base64urlencode(Playload),
        secret-key
    )
    */


    //-----------------------------------------------------------------------------------------

    public static final String PARAM_ROLES = "roles";
    public static final String PARAM_PERMS = "perms";

    //-----------------------------------------------------------------------------------------

    public static String issueJwt(JwtPayload payload, String secretkey)
    {
        JwtBuilder jwt  =  Jwts.builder();

        //
        jwt.setHeaderParam("typ", "JWT");
        jwt.setId(payload.getId());
        jwt.setSubject(payload.getSubject());// 主题(用户)
        jwt.setIssuer(payload.getIssuer());//签发者
        jwt.setAudience(payload.getAudience()); //接收者
        jwt.setIssuedAt(payload.getIssuedAt());//签发时间
        jwt.setExpiration(payload.getExpiration());//有效时间

        //
        if(StringUtils.NotEmpty(payload.getRoles())) jwt.claim(PARAM_ROLES, payload.getRoles());//访问主张-角色
        if(StringUtils.NotEmpty(payload.getPerms())) jwt.claim(PARAM_PERMS, payload.getPerms());//访问主张-权限

        //
        jwt.compressWith(CompressionCodecs.DEFLATE);//压缩，可选GZIP
        jwt.signWith(SignatureAlgorithm.HS512, secretkey);//秘钥

        return jwt.compact();
    }

    public static JwtPayload parseJwt(String jwt, String secretkey)
    {
        JwtPayload payload;
        try
        {
            Claims claims = Jwts.parser()
                .setSigningKey(secretkey)
                .parseClaimsJws(jwt)
                .getBody();

            //
            payload = new JwtPayload();
            payload.setId(claims.getId());
            payload.setSubject(claims.getSubject());// 主题(用户)
            payload.setIssuer(claims.getIssuer());// 签发者
            payload.setIssuedAt(claims.getIssuedAt());// 签发时间
            payload.setAudience(claims.getAudience());// 接收方
            payload.setExpiration(claims.getExpiration()); //过期时间
            payload.setRoles(claims.get(PARAM_ROLES, String.class));// 访问主张-角色
            payload.setPerms(claims.get(PARAM_PERMS, String.class));// 访问主张-权限
        }
        catch (ExpiredJwtException e)
        {
            throw new RestAuthorizeException("JWT 令牌过期:" + e.getMessage());
        }
        catch (UnsupportedJwtException e)
        {
            throw new RestAuthorizeException("JWT 令牌无效:" + e.getMessage());
        }
        catch (MalformedJwtException e)
        {
            throw new RestAuthorizeException("JWT 令牌格式错误:" + e.getMessage());
        }
        catch (SignatureException e)
        {
            throw new RestAuthorizeException("JWT 令牌签名无效:" + e.getMessage());
        }
        catch (IllegalArgumentException e)
        {
            throw new RestAuthorizeException("JWT 令牌参数异常:" + e.getMessage());
        }
        catch (Exception e)
        {
            throw new RestAuthorizeException("JWT 令牌错误:" + e.getMessage());
        }

        return payload;
    }

    //-----------------------------------------------------------------------------------------

}
