/*
 * Id$: zuv-cloud:z-web-support:cc.zuv.web.support.aspect.SysLogAspect:20181226101928
 *
 * SysLogAspect.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.web.support.aspect;

import cc.zuv.lang.StringUtils;
import cc.zuv.utility.KeyGenUtils;
import cc.zuv.web.support.annotation.SysLogRender;
import cc.zuv.web.support.enums.SysLogLevel;
import cc.zuv.web.support.enums.SysLogModule;
import cc.zuv.web.support.utility.WebsPathUtils;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.http.MediaType;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;

/**
 * 系统日志切面处理
 *
 * @author			Kama Luther
 * @version			0.1
 * @since           0.1
 * @create.date     2014-2-11 下午02:19:48
 * @modify.date     2014-2-11 下午02:19:48
 */
@Slf4j
public abstract class SysLogAspect
{

    //-----------------------------------------------------------------------------------------

	@Pointcut("@annotation(cc.zuv.web.support.annotation.SysLogRender)")
	public void logPointCut()
    {

	}

	@Around("logPointCut()")
	public Object around(ProceedingJoinPoint point) throws Throwable
    {
        //开始时间
		long beginTime = System.currentTimeMillis();

		//执行方法
        Object result = null;
        Exception exception = null;
        try
        {
            result = point.proceed();
        }
        catch (Exception e)
        {
            exception = e;
        }


		//执行时长(毫秒)
		long elapse = System.currentTimeMillis() - beginTime;

		//保存日志
		saveSysLog(point, elapse, result, exception);

		return result;
	}

    //-----------------------------------------------------------------------------------------

	private void saveSysLog(ProceedingJoinPoint joinPoint, long elapse, Object result, Exception exception)
    {
        //
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();

        //
		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		Method method = signature.getMethod();
        SysLogRender syslogrender = method.getAnnotation(SysLogRender.class);

        if(syslogrender==null) return;

        //注解的内容
        SysLogModule    _module     = syslogrender.module();
        SysLogLevel     _level      = syslogrender.level();
        String          _title      = syslogrender.value();
        String          _ident      = syslogrender.ident();
        String          _modulestr  = syslogrender.modulestr();
        String          _levelstr   = syslogrender.levelstr();

		//请求的方法
		String cname = joinPoint.getTarget().getClass().getName();
		String mname = signature.getName();
        String _method = cname + "." + mname + "()";

		//请求的参数
        String _params = null;
        String contentType = request.getContentType();
        if (contentType == null
            || contentType.contains(MediaType.APPLICATION_FORM_URLENCODED_VALUE)
            || contentType.contains(MediaType.APPLICATION_JSON_VALUE))
        {
            Object[] args = joinPoint.getArgs();
            if(args!=null && args.length>0)
            {
                Object[] newargs = new Object[args.length];

                int i = 0;
                for(Object arg : args)
                {
                    if(arg instanceof ServletResponse || arg instanceof ServletRequest)
                    {
                        continue;
                    }
                    newargs[i++] = arg;
                }

                _params = new Gson().toJson(newargs);
            }
        }

        //请求的地址
        String _hostip = WebsPathUtils.getClientIp(request);
        String _requri = request.getRequestURI();
        String _reqmtd = request.getMethod();

        //多个IP取第一个
        if(_hostip!=null && _hostip.contains(","))
        {
            _hostip = _hostip.split(",")[0];
        }

        //
        String _result = (result==null
            || result instanceof ServletResponse
            || result instanceof ServletRequest
            || result instanceof OutputStream
            || result instanceof InputStream
            || result instanceof File)?null:new Gson().toJson(result);
        String _exception = exception!=null?exception.getMessage():"OK";

        //
        if(StringUtils.NotEmpty(_levelstr) && StringUtils.NotEmpty(_modulestr))
        {
            saveSysLog(_levelstr, _modulestr, _title, _ident, _method, _params, _hostip, _requri, _reqmtd, elapse, _result, _exception);
        }
        else
        {
            saveSysLog(_level, _module, _title, _ident, _method, _params, _hostip, _requri, _reqmtd, elapse, _result, _exception);
        }
	}

    protected abstract void saveSysLog(SysLogLevel level, SysLogModule module, String title, String ident,
                                       String method, String params, String hostip,
                                       String requri, String reqmtd,
                                       long elapse, String result, String exception);

    protected abstract void saveSysLog(String level, String module, String title, String ident,
                                       String method, String params, String hostip,
                                       String requri, String reqmtd,
                                       long elapse, String result, String exception);

    //-----------------------------------------------------------------------------------------

    private static final String REQUEST_ID = "$request_id";

    @Before("logPointCut()")
    public void doBefore(JoinPoint joinPoint) throws Throwable
    {
        //
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();

        //
        String requestid = KeyGenUtils.uuid();
        request.setAttribute(REQUEST_ID, requestid);

        //
        String sessionid = request.getSession().getId();
        String requesturi = request.getRequestURI();

        //
        log.info("request: id={}, session= {}, requesturi={}", requestid, sessionid, requesturi);
    }

    @AfterReturning(returning = "ret", pointcut = "logPointCut()")
    public void doAfterReturning(Object ret) throws Throwable
    {
        //
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();

        //
        String requestid = (String) request.getAttribute(REQUEST_ID);
        String sessionid = request.getSession().getId();
        String retrunval = new Gson().toJson(ret);

        //
        log.info("response: id={}, session= {}, ret={}", requestid, sessionid, retrunval);
    }

    @AfterThrowing(pointcut = "logPointCut()", throwing = "e")
    public void doAfterThrowing(Exception e)
    {
        //
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();

        //
        String requestid = (String) request.getAttribute(REQUEST_ID);
        String sessionid = request.getSession().getId();
        String message = e.getMessage();

        //
        log.info("throwing: id={}, session= {}, message={}", requestid, sessionid, message);
    }

    //-----------------------------------------------------------------------------------------


    //-----------------------------------------------------------------------------------------


}
