/*
 * Id$: zuv-cloud:z-web-support:cc.zuv.web.support.exception.RestExceptionHandler:20181225110811
 *
 * RestExceptionHandler.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.web.support.exception;

import cc.zuv.IERRCode;
import cc.zuv.lang.StringUtils;
import cc.zuv.web.support.IWebERRCode;
import cc.zuv.web.support.payload.RestResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.servlet.http.HttpServletRequest;
import java.util.Locale;

/**
 * File Description
 *
 * @author			Kama Luther
 * @version			0.1
 * @since           0.1
 * @create.date     2014-2-11 下午02:19:48
 * @modify.date     2014-2-11 下午02:19:48
 */
@Slf4j
@RestControllerAdvice
public class RestExceptionHandler
{

    //-----------------------------------------------------------------------------------------

    @Autowired
    private MessageSource messageSource;

    public String getMessage(String messageKey, Locale locale, Object... params)
    {
        return messageSource.getMessage(messageKey, params, locale);
    }

    public String getMessage(String messageKey, Object... params)
    {
        Locale locale = LocaleContextHolder.getLocale();
        return messageSource.getMessage(messageKey, params, locale);
    }

    //-----------------------------------------------------------------------------------------

	@ExceptionHandler(RestException.class)
	public RestResult handleRestException(HttpServletRequest req, RestException e)
    {
        IWebERRCode errcode = e.getErrcode();
        String      message = e.getMessage();
        if(message==null || message.trim().length()==0)
        {
            Locale locale = LocaleContextHolder.getLocale(); //后续可以动态调整
            message = getMessage(errcode.name(), locale, e.getParams());
        }

        return getRestResult(errcode, message, req);
	}

    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
	@ExceptionHandler(RestDatabaseException.class)
	public RestResult handleDatabaseException(HttpServletRequest req, RestDatabaseException e)
    {
        return getRestResult(HttpStatus.INTERNAL_SERVER_ERROR, IERRCode.ERRCODE_DATABASE, e.getMessage(), req);
	}

    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(RestIOException.class)
    public RestResult handleIOException(HttpServletRequest req, RestIOException e)
    {
        return getRestResult(HttpStatus.INTERNAL_SERVER_ERROR, IERRCode.ERRCODE_IO, e.getMessage(), req);
    }

    @ResponseStatus(HttpStatus.NOT_FOUND)
    @ExceptionHandler(RestNotFoundException.class)
    public RestResult handleNotFoundException(HttpServletRequest req, RestNotFoundException e)
    {
        return getRestResult(HttpStatus.NOT_FOUND, IERRCode.ERRCODE_RESOURCE, e.getMessage(), req);
    }

    @ResponseStatus(HttpStatus.CONFLICT)
    @ExceptionHandler(RestHasExistsException.class)
    public RestResult handleHasExistsException(HttpServletRequest req, RestHasExistsException e)
    {
        return getRestResult(HttpStatus.CONFLICT, IERRCode.ERRCODE_RESOURCE, e.getMessage(), req);
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ExceptionHandler(RestBadRequestException.class)
    public RestResult handleBadRequestException(HttpServletRequest req, RestBadRequestException e)
    {
        return getRestResult(HttpStatus.BAD_REQUEST, IERRCode.ERRCODE_REQUEST, e.getMessage(), req);
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public RestResult handleArgNotValidException(HttpServletRequest req, MethodArgumentNotValidException e)
    {
        BindingResult result = e.getBindingResult();

        StringBuilder sb = new StringBuilder();
        for (FieldError error : result.getFieldErrors())
        {
            String message = String.format("%s:%s ", error.getField(), error.getDefaultMessage());
            sb.append(message).append("\n");
        }

        return getRestResult(HttpStatus.BAD_REQUEST, IERRCode.ERRCODE_VALIDATE, sb.toString(), req);
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public RestResult handleMethodNotSupportException(HttpServletRequest req, HttpRequestMethodNotSupportedException e)
    {
        return getRestResult(HttpStatus.BAD_REQUEST, IERRCode.ERRCODE_REQUEST, "方法不支持", req);
    }

    @ResponseStatus(HttpStatus.UNAUTHORIZED)
	@ExceptionHandler(RestAuthorizeException.class)
	public RestResult handleAuthorizationException(HttpServletRequest req, RestAuthorizeException e)
    {
        return getRestResult(HttpStatus.UNAUTHORIZED, IERRCode.ERRCODE_AUTHORIZE, e.getMessage(), req);
	}

    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
	@ExceptionHandler(Exception.class)
	public RestResult handleException(HttpServletRequest req, Exception e)
    {
		log.error(e.getMessage(), e);
        return getRestResult(HttpStatus.INTERNAL_SERVER_ERROR, IERRCode.ERRCODE_FAILURE, e.getMessage(), req);
	}

    //-----------------------------------------------------------------------------------------

    public RestResult<String> getRestResult(HttpStatus _status, int _errcode, String _message, HttpServletRequest _req)
    {
        String      epoint  = _req.getRequestURI();
        String      method  = _req.getMethod();
        String      message = StringUtils.NotEmpty(_message)?_message:_status.getReasonPhrase();
        return new RestResult<>(_status, epoint, method, _errcode, message);
    }

    public RestResult<String> getRestResult(IWebERRCode _errcode, String _message, HttpServletRequest _req)
    {
        Locale locale = LocaleContextHolder.getLocale(); //后续可以动态调整
        String      epoint  = _req.getRequestURI();
        String      method  = _req.getMethod();
        HttpStatus  status  = _errcode.status();
        int         errcode = _errcode.errcode();
        String      message = StringUtils.NotEmpty(_message)?_message:getMessage(_errcode.name(), locale);
        return new RestResult<>(status, epoint, method, errcode, message);
    }

    //-----------------------------------------------------------------------------------------

}
