/*
 * Id$: zuv-cloud:z-web-support:cc.zuv.web.support.utility.HttpRenderUtils:20190425154634
 *
 * HttpRenderUtils.java
 * Copyright (c) 2002-2020 Luther Inc.
 * http://zuv.cc
 * All rights reserved.
 */

package cc.zuv.web.support.utility;

import cc.zuv.ios.Resourcer;
import cc.zuv.lang.StringUtils;
import cc.zuv.utility.CodecUtils;
import cc.zuv.utility.MimeUtils;
import cc.zuv.web.support.payload.RestResult;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;

import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.awt.image.BufferedImage;
import java.io.*;

/**
 * zuv-cloud File Description
 *
 * @author          Kama Luther
 * @version         0.1
 * @since           0.1
 * @create.date     2019-04-25 11:47
 * @modify.date     2019-04-25 11:47
 */
@Slf4j
public class HttpRenderUtils
{

    //-----------------------------------------------------------------------------------------

    public static boolean require_json(HttpServletRequest request)
    {
        String contentType = request.getContentType();
        return (contentType != null && contentType.toLowerCase().contains(MediaType.APPLICATION_JSON_VALUE));
    }

    public static boolean require_xml(HttpServletRequest request)
    {
        String contentType = request.getContentType();
        return (contentType != null && contentType.toLowerCase().contains(MediaType.APPLICATION_XML_VALUE));
    }

    //-----------------------------------------------------------------------------------------

    public static void render_json(HttpServletRequest request, HttpServletResponse response,
                                   int status, int errcode, String message)
        throws IOException
    {
        String      epoint  = request.getRequestURI();
        String      method  = request.getMethod();
        RestResult<String> result = new RestResult<>(status, epoint, method, errcode, message);
        render_content(response, status, MimeUtils.FILE_TYPE_JSON, result);
    }

    public static void render_xml(HttpServletRequest request, HttpServletResponse response,
                                   int status, int errcode, String message)
        throws IOException
    {
        String      epoint  = request.getRequestURI();
        String      method  = request.getMethod();
        RestResult<String> result = new RestResult<>(status, epoint, method, errcode, message);
        render_content(response, status, MimeUtils.FILE_TYPE_XML, result);
    }

    //-----------------------------------------------------------------------------------------

    public static void render_html(HttpServletResponse response, String html)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_HTML, html);
    }

    public static void render_text(HttpServletResponse response, String txt)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_TEXT, txt);
    }

    public static void render_xml(HttpServletResponse response, String xml)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_XML, xml);
    }

    public static void render_json(HttpServletResponse response, String json)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_JSON, json);
    }

    public static void render_markdown(HttpServletResponse response, String md)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_MARKD, md);
    }

    public static void render_content(HttpServletResponse response, String mime, String content)
        throws IOException
    {
        response.setContentType(mime);
        PrintWriter writer = response.getWriter();
        writer.println(content);
        writer.close();
    }

    public static void render_content(HttpServletResponse response, int status, String mime, String content)
        throws IOException
    {
        response.setStatus(status);
        response.setContentType(mime);
        PrintWriter writer = response.getWriter();
        writer.println(content);
        writer.close();
    }

    //-----------------------------------------------------------------------------------------

    public static void render_html(HttpServletResponse response, RestResult rest)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_HTML, rest);
    }

    public static void render_text(HttpServletResponse response, RestResult rest)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_TEXT, rest);
    }

    public static void render_xml(HttpServletResponse response, RestResult rest)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_XML, rest);
    }

    public static void render_json(HttpServletResponse response, RestResult rest)
        throws IOException
    {
        render_content(response, MimeUtils.FILE_TYPE_JSON, rest);
    }

    public static void render_content(HttpServletResponse response, String mime, RestResult rest)
        throws IOException
    {
        response.setContentType(mime);
        PrintWriter writer = response.getWriter();
        writer.println(new Gson().toJson(rest));
        writer.close();
    }

    public static void render_content(HttpServletResponse response, int status, String mime, RestResult rest)
        throws IOException
    {
        response.setStatus(status);
        response.setContentType(mime);
        PrintWriter writer = response.getWriter();
        writer.println(new Gson().toJson(rest));
        writer.close();
    }

    //-----------------------------------------------------------------------------------------

    public static void redirect(HttpServletResponse response, String uri)
        throws IOException
    {
        response.sendRedirect(uri); //response.setHeader("Location", uri);
    }

    public static void render_mime(HttpServletResponse response, String filename)
    {
        String mime = MimeUtils.queryMimeByFileName(filename);
        response.setContentType(StringUtils.IsEmpty(mime)?MimeUtils.FILE_TYPE_OCTET:mime);
    }

    public static void render_mime(HttpServletResponse response, String filename, File file)
        throws IOException
    {
        String mime = MimeUtils.queryMimeByFileName(filename);
        response.setContentType(StringUtils.IsEmpty(mime)?MimeUtils.FILE_TYPE_OCTET:mime);

        OutputStream os = response.getOutputStream();
        Resourcer.copy(file, os);
    }

    public static void download_mime(HttpServletResponse response, String filename)
    {
        String disp = String.format("attachment; filename=%s", CodecUtils.url_encode(filename));
        response.setContentType(MimeUtils.FILE_TYPE_OCTET);
        response.setHeader("Content-Disposition", disp);
    }

    public static void download_mime(HttpServletResponse response, String filename, File file)
        throws IOException
    {
        String disp = String.format("attachment; filename=%s", CodecUtils.url_encode(filename));
        response.setContentType(MimeUtils.FILE_TYPE_OCTET);
        response.setHeader("Content-Disposition", disp);

        OutputStream os = response.getOutputStream();
        Resourcer.copy(file, os);
    }

    public static void download_mime(HttpServletResponse response, String filename, InputStream is)
        throws IOException
    {
        String disp = String.format("attachment; filename=%s", CodecUtils.url_encode(filename));
        response.setContentType(MimeUtils.FILE_TYPE_OCTET);
        response.setHeader("Content-Disposition", disp);

        OutputStream os = response.getOutputStream();
        Resourcer.copy(is, os);
    }

    //-----------------------------------------------------------------------------------------

    public static void render_image(HttpServletResponse response, BufferedImage image)
        throws IOException
    {
        response.setHeader("Cache-Control", "no-store, no-cache");
        response.setHeader("Pragma", "no-cache");
        response.setDateHeader("Expires", 0);
        response.setContentType("image/jpeg");
        ServletOutputStream out = response.getOutputStream();
        ImageIO.write(image, "jpg", out);
        out.close();
    }


    //-----------------------------------------------------------------------------------------

    public static void render_redirect(HttpServletResponse response, String url)
        throws IOException
    {
        response.sendRedirect(url);
    }

    //-----------------------------------------------------------------------------------------

    public static String verbose(HttpServletRequest request, HttpServletResponse response, boolean isdebug, String sessionname, String cookiename)
    {
        //
        StringBuilder sb = new StringBuilder();

        //--------------------------------------------------------------
        // Session
        //--------------------------------------------------------------

        HttpSession session = request.getSession(false);
        if (session == null)
            session = request.getSession(true);

        session.setMaxInactiveInterval(2000);
        String  sessid      = session.getId();
        boolean isnew 		= session.isNew();
        long 	createtime 	= session.getCreationTime();
        long 	laccesstime = session.getLastAccessedTime();
        int 	maxinactive = session.getMaxInactiveInterval();
        log.info("[Session] : ID="+sessid+",New="+isnew+",Create=" + createtime + ",Access=" + laccesstime+",Inactive="+maxinactive);

        //校验Session
        if(session.getAttribute(sessionname) != null)
        {
            boolean session_visited = (Boolean)session.getAttribute(sessionname);
            log.info("[Session] : " + session_visited);
        }

        if(isdebug)
        {
            session.setAttribute(sessionname, true);
        }

        sb.append("[Session] : ").append(sessid).append("<br/>");

        //--------------------------------------------------------------
        // Cookie
        //--------------------------------------------------------------

        //校验Cookie
        String cookie_visited = HttpCookieUtils.getCookie(request, cookiename);
        log.info("[Cookie] : " + cookie_visited);

        if(isdebug)
        {
            HttpCookieUtils.setCookie(response, cookiename, "true", null);
        }

        sb.append("[Cookie] : ").append(cookie_visited).append("<br/>");

        //--------------------------------------------------------------
        // Header
        //--------------------------------------------------------------

        //
        String referer      = request.getHeader("referer");
        String useragent    = request.getHeader("user-agent");

        /*String proxyaddr1   = request.getHeader("x-real-ip");       //ngix
        String proxyaddr2   = request.getHeader("x-forwarded-for"); //apache
        String remoteaddr   = request.getRemoteAddr();

        //校验IP地址
        String proxyaddr    = (proxyaddr2!=null && proxyaddr2.length()>0)?proxyaddr2:proxyaddr1;
        String address      = (proxyaddr!=null && proxyaddr.length()>0)?proxyaddr:remoteaddr;*/

        String address      = HttpRequestUtils.getClientIp(request);
        log.info("[Header] : " + referer + "," + useragent + "," + address);

        sb.append("[Header] <br/>");
        sb.append("referer : ").append(referer).append("<br/>");
        sb.append("useragent : ").append(useragent).append("<br/>");
        sb.append("address : ").append(address).append("<br/>");

        //--------------------------------------------------------------
        // Request
        //--------------------------------------------------------------

        // content为webapp目录
        // http://zuv.cc:8080/content/manage/user_detail?userid=12

        String urls     = request.getRequestURL().toString();   // http://zuv.cc:8080/content/manage/user_detail
        String uris     = request.getRequestURI();              // /content/manage/user_detail
        String path     = request.getServletPath();             // /manage/user_detail
        String para     = request.getQueryString();             // userid=12
        String content  = request.getContextPath();             // /content
        String selfurl  = (para != null) ? (urls + "?" + para) : urls;    // http://zuv.cc:8080/content/manage/user_detail?userid=12
        log.info("[Request] : " + urls + "," + uris + "," + path + "," + content + "," + selfurl);

        sb.append("[Request] <br/>");
        sb.append("urls : ").append(urls).append("<br/>");
        sb.append("uris : ").append(uris).append("<br/>");
        sb.append("path : ").append(path).append("<br/>");
        sb.append("para : ").append(para).append("<br/>");
        sb.append("content : ").append(content).append("<br/>");
        sb.append("selfurl : ").append(selfurl).append("<br/>");

        //--------------------------------------------------------------

        return sb.toString();
    }

    //-----------------------------------------------------------------------------------------

}
