package cd.connect.jersey.common;

import cd.connect.app.config.ConfigKey;
import cd.connect.app.config.DeclaredConfigResolver;

import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.ContainerResponseContext;
import javax.ws.rs.container.ContainerResponseFilter;
import javax.ws.rs.container.PreMatching;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.Provider;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

// from https://stackoverflow.com/questions/28065963/how-to-handle-cors-using-jax-rs-with-jersey

@Provider
@PreMatching
public class CorsFilter implements ContainerRequestFilter, ContainerResponseFilter {
  @ConfigKey("jersey.cors.headers")
  List<String> allowedHeaders = Arrays.asList(
    // Whatever other non-standard/safe headers (see list above)
    // you want the client to be able to send to the server,
    // put it in this list. And remove the ones you don't want.
    "X-Requested-With", "Authorization", "Content-type",
      "Accept-Version", "Content-MD5", "CSRF-Token");

  private String headers;

  public CorsFilter() {
    DeclaredConfigResolver.resolve(this);

    headers = String.join(",", allowedHeaders);
  }

  /**
   * Method for ContainerRequestFilter.
   */
  @Override
  public void filter(ContainerRequestContext request) throws IOException {

    // If it's a preflight request, we abort the request with
    // a 200 status, and the CORS headers are added in the
    // response filter method below.
    if (isPreflightRequest(request)) {
      request.abortWith(Response.ok().build());
      return;
    }
  }

  /**
   * A preflight request is an OPTIONS request
   * with an Origin header.
   */
  private static boolean isPreflightRequest(ContainerRequestContext request) {
    return request.getHeaderString("Origin") != null
      && request.getMethod().equalsIgnoreCase("OPTIONS");
  }

  /**
   * Method for ContainerResponseFilter.
   */
  @Override
  public void filter(ContainerRequestContext request, ContainerResponseContext response) {

    // if there is no Origin header, then it is not a
    // cross origin request. We don't do anything.
    if (request.getHeaderString("Origin") == null) {
      return;
    }

    // If it is a preflight request, then we add all
    // the CORS headers here.
    if (isPreflightRequest(request)) {
      response.getHeaders().add("Access-Control-Allow-Credentials", "true");
      response.getHeaders().add("Access-Control-Allow-Methods",
        "GET, POST, PUT, DELETE, OPTIONS, HEAD");
      response.getHeaders().addAll("Access-Control-Allow-Headers", headers);
    }

    // Cross origin requests can be either simple requests
    // or preflight request. We need to add this header
    // to both type of requests. Only preflight requests
    // need the previously added headers.
    response.getHeaders().add("Access-Control-Allow-Origin", "*");
  }
}
