package cloud.filibuster.junit.assertions.protocols;

import cloud.filibuster.exceptions.filibuster.FilibusterUnsupportedByHTTPServerException;
import cloud.filibuster.junit.assertions.BlockType;
import cloud.filibuster.junit.assertions.protocols.GenericAssertions;
import cloud.filibuster.junit.server.core.FilibusterCore;
import com.google.protobuf.GeneratedMessage;
import com.google.protobuf.GeneratedMessageV3;
import io.grpc.MethodDescriptor;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;
import org.junit.jupiter.api.function.ThrowingConsumer;

import static cloud.filibuster.instrumentation.helpers.Property.getServerBackendCanInvokeDirectlyProperty;
import static cloud.filibuster.junit.assertions.Helpers.incrementTestScopeCounter;

public class GrpcAssertions extends GenericAssertions {
    /**
     * Execute a test and catch GRPC exceptions.
     *
     * @param block block to execute containing test.
     * @param catchBlock assertion block triggered, if and only if, GRPC exception is thrown
     * @throws Throwable rethrown exceptions generated by either the @{block} or @{catchBlock}
     */
    public static void tryGrpcAndCatchGrpcExceptions(
            Runnable block,
            ThrowingConsumer<Throwable> catchBlock
    ) throws Throwable {
        // Increment the scope counter for entry into this block.
        incrementTestScopeCounter(BlockType.TEST);

        try {
            // Run the provided test block.
            block.run();
        } catch (Throwable t) {

            if (!wasFaultInjected()) {
                // If a fault wasn't injected -- which returns false always when Filibuster is disabled --
                // then, rethrow whatever exception occurred.
                //
                // It's unclear why the compiler analysis can't determine that this function throws a Throwable,
                // since the default implementation of this function returns true.
                //
                throw t;
            }

            // Fault injected.

            // This warning is not true, ignore.
            if (t instanceof StatusException || t instanceof StatusRuntimeException) {
                // If a GRPC exception was thrown, trigger the catch block.
                catchBlock.accept(t);
            } else {
                // Otherwise, it might be a normal assertion failure.
                // If so, rethrow.
                throw t;
            }
        }
    }

    /**
     * Returns true if a fault was injected on a GRPC service.
     *
     * @param serviceName GRPC fully-qualified service name including namespace.
     * @return if a fault was injected
     */
    public static boolean wasFaultInjectedOnService(String serviceName) {
        if (getServerBackendCanInvokeDirectlyProperty()) {
            if (FilibusterCore.hasCurrentInstance()) {
                return FilibusterCore.getCurrentInstance().wasFaultInjectedOnService(serviceName);
            } else {
                return false;
            }
        } else {
            return wasFaultInjectedHelper("/filibuster/fault-injected/service/" + serviceName);
        }
    }

    /**
     * Returns true if a fault was injected on a GRPC method.
     *
     * @param method GRPC method descriptor
     * @return if a fault was injected.
     */
    public static boolean wasFaultInjectedOnMethod(MethodDescriptor method) {
        String fullyQualifiedMethodName = method.getFullMethodName();

        if (getServerBackendCanInvokeDirectlyProperty()) {
            String[] split = fullyQualifiedMethodName.split("/", 2);

            if (FilibusterCore.hasCurrentInstance()) {
                return FilibusterCore.getCurrentInstance().wasFaultInjectedOnMethod(split[0], split[1]);
            } else {
                return false;
            }
        } else {
            return wasFaultInjectedHelper("/filibuster/fault-injected/method/" + fullyQualifiedMethodName);
        }
    }

    /**
     * Returns true if a fault was injected on a GRPC request.
     *
     * @param request GRPC message
     * @return if a fault was injected.
     */
    public static boolean wasFaultInjectedOnRequest(GeneratedMessage request) {
        if (getServerBackendCanInvokeDirectlyProperty()) {
            if (FilibusterCore.hasCurrentInstance()) {
                return FilibusterCore.getCurrentInstance().wasFaultInjectedOnRequest(request.toString());
            } else {
                return false;
            }
        } else {
            throw new FilibusterUnsupportedByHTTPServerException("wasFaultInjectedOnRequest only supported with local server.");
        }
    }

    /**
     * Returns true if a fault was injected on a GRPC request.
     *
     * @param request GRPC message
     * @return if a fault was injected.
     */
    public static boolean wasFaultInjectedOnRequest(GeneratedMessageV3 request) {
        if (getServerBackendCanInvokeDirectlyProperty()) {
            if (FilibusterCore.hasCurrentInstance()) {
                return FilibusterCore.getCurrentInstance().wasFaultInjectedOnRequest(request.toString());
            } else {
                return false;
            }
        } else {
            throw new FilibusterUnsupportedByHTTPServerException("wasFaultInjectedOnRequest only supported with local server.");
        }
    }
}
