package cn.callmee.springboot.pulsar.starter.client.config;

import cn.callmee.springboot.pulsar.starter.client.annotations.PulsarConsumer;
import cn.callmee.springboot.pulsar.starter.client.annotations.PulsarProducer;
import cn.callmee.springboot.pulsar.starter.client.domain.PulsarProducerTemplate;
import cn.callmee.springboot.pulsar.starter.client.enums.Serialization;
import cn.callmee.springboot.pulsar.starter.client.exceptions.ClientInitException;
import cn.callmee.springboot.pulsar.starter.client.exceptions.ConsumerInitException;
import cn.callmee.springboot.pulsar.starter.client.exceptions.ProducerInitException;
import cn.callmee.springboot.pulsar.starter.client.holder.ConsumerHolder;
import cn.callmee.springboot.pulsar.starter.client.holder.ProducerHolder;
import cn.callmee.springboot.pulsar.starter.client.message.FailedMessage;
import cn.callmee.springboot.pulsar.starter.client.utils.SchemaUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.pulsar.client.api.*;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.DependsOn;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Slf4j
@DependsOn({"pulsarClient"})
public class PulsarClientInitialConfiguration extends PulsarClientInitial {

    public PulsarClientInitialConfiguration() {
        log.debug("[CONFIG]初始化 Pulsar 消息队列");
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) {
        return bean;
    }

    @Override
    protected void postInitializationConsumer(Method $, Object bean, Class<?> beanClass) {
        PulsarConsumer pulsarConsumer = $.getAnnotation(PulsarConsumer.class);
        String consumerName = pulsarUrlGenerator.buildConsumerName(beanClass, $);
        ConsumerHolder consumerHolder = new ConsumerHolder(
                pulsarConsumer, $, bean, SchemaUtils.getParameterType($, pulsarConsumer.paramName())
        );
        if (pulsarConsumer.autoStart()) {
            Consumer<?> consumer = buildConsumer(consumerName, consumerHolder);
            if (null != consumer) {
                COLLECT_CONSUMERS.add(
                        consumer
                );
            }
        }
        CONSUMERS.put(consumerName, consumerHolder);
        addInLog(
                REG_CONSUMER_LOG_ARRAY,
                "- topic: {}\n  handler: {}.{}\n  interceptor: {}",
                new Object[]{pulsarConsumer.topic(), beanClass.getName(), $.getName(), consumerHolder.getInterceptor()}
        );
    }

    @Override
    protected void postInitializationProducer(Method $, Class<?> beanClass) {
        String name = $.getName();
        PulsarProducer annotationPulsarProducer = $.getAnnotation(PulsarProducer.class);
        String[] annotationTopicArr = SchemaUtils.generateProducerTopicList(annotationPulsarProducer);
        for (String annotationTopic : annotationTopicArr) {
            String topic = STRING_VALUE_RESOLVER.resolveStringValue(StringUtils.hasText(annotationTopic) ? annotationTopic : name);
            Class<?> returnType = $.getReturnType();
            Serialization producerTripleMiddle = annotationPulsarProducer.serialization();
            ProducerHolder holder = new ProducerHolder(
                    topic,
                    returnType,
                    producerTripleMiddle,
                    annotationPulsarProducer.namespace()
            );
            PRODUCERS.put(
                    topic,
                    this.buildProducer(
                            holder
                    )
            );
            addInLog(
                    REG_PRODUCER_LOG_ARRAY,
                    "- topic: {}\n  handler: {}.{}\n  msgType: {}\n  interceptor: {}",
                    new Object[]{
                            topic, beanClass.getName(), name, returnType, holder.getInterceptor()
                    }
            );
        }
    }

    @SneakyThrows
    private Producer<?> buildProducer(ProducerHolder holder) {
        try {
            final ProducerBuilder<?> producerBuilder = pulsarClient.newProducer(getSchema(holder))
                    .blockIfQueueFull(true)
                    .topic(holder.getNamespace()
                            .map(namespace -> pulsarUrlGenerator.buildTopicUrl(holder.getTopic(), namespace))
                            .orElseGet(() -> pulsarUrlGenerator.buildTopicUrl(holder.getTopic())));

            if (pulsarProperties.isAllowInterceptor()) {
                // producerBuilder.intercept(producerInterceptor);
                producerBuilder.intercept(holder.getInterceptor().getConstructor().newInstance());
            }

            return producerBuilder.create();
        } catch (PulsarClientException e) {
            throw new ProducerInitException("Failed to init producer.", e);
        }
    }

    @SneakyThrows
    private Consumer<?> buildConsumer(String generatedConsumerName, ConsumerHolder holder) {
        try {
            final String consumerName = PulsarClientInitialConfiguration.STRING_VALUE_RESOLVER.resolveStringValue(holder.getAnnotation().consumerName());
            final String subscriptionName = PulsarClientInitialConfiguration.STRING_VALUE_RESOLVER.resolveStringValue(holder.getAnnotation().subscriptionName());
            final String topicName = PulsarClientInitialConfiguration.STRING_VALUE_RESOLVER.resolveStringValue(holder.getAnnotation().topic());
            final String namespace = PulsarClientInitialConfiguration.STRING_VALUE_RESOLVER.resolveStringValue(holder.getAnnotation().namespace());
            final SubscriptionType subscriptionType = pulsarUrlGenerator.getSubscriptionType(holder);
            // 反射方法中的形参并根据注解的字段进行过滤
            List<Parameter> invokeParameterStructure = SchemaUtils.getInvokeParameterStructure(holder);
            final ConsumerBuilder<?> consumerBuilder = pulsarClient
                    .newConsumer(
                            SchemaUtils.getSchema(
                                    holder.getAnnotation().serialization(),
                                    // 优先通过反射得出的ConsumerHolder的信息进行消费者绑定
                                    null != holder.getType() ? holder.getType() : holder.getAnnotation().clazz()
                            )
                    )
                    .consumerName(pulsarUrlGenerator.buildPulsarConsumerName(consumerName, generatedConsumerName))
                    .subscriptionName(pulsarUrlGenerator.buildPulsarSubscriptionName(subscriptionName, generatedConsumerName))
                    .topic(pulsarUrlGenerator.buildTopicUrl(topicName, namespace))
                    .subscriptionType(subscriptionType)
                    .subscriptionInitialPosition(holder.getAnnotation().initialPosition())
                    .messageListener((consumer, msg) -> {
                        log.info("handle consumer: {}", consumer.getConsumerName());
                        try {
                            final Method method = holder.getHandler();
                            method.setAccessible(true);

                            Object args;
                            if (holder.isWrapped()) {
                                args = wrapMessage(msg);
                            } else {
                                args = msg.getValue();
                            }
                            // 根据方法形参结构装配消费执行方法的实参
                            Object[] objects = invokeParameterStructure.stream().map(
                                    i -> null == i ? null : args
                            ).toArray();
                            // 使用反射执行方法
                            method.invoke(holder.getBean(), objects);

                            consumer.acknowledge(msg);
                        } catch (Exception e) {
                            log.warn("consume failed: {}", e.getLocalizedMessage());
                            consumer.negativeAcknowledge(msg);
                            SINK.tryEmitNext(new FailedMessage(e, consumer, msg));
                        }
                    });

            if (pulsarProperties.isAllowInterceptor()) {
                // consumerBuilder.intercept(consumerInterceptor);
                consumerBuilder.intercept(holder.getInterceptor().getConstructor().newInstance());
            }

            if (pulsarProperties.getConsumer().getAckTimeoutMs() > 0) {
                consumerBuilder.ackTimeout(pulsarProperties.getConsumer().getAckTimeoutMs(), TimeUnit.MILLISECONDS);
            }

            pulsarUrlGenerator.buildDeadLetterPolicy(
                    holder.getAnnotation().maxRedeliverCount(),
                    holder.getAnnotation().deadLetterTopic(),
                    consumerBuilder);

            return consumerBuilder.subscribe();
        } catch (PulsarClientException | ClientInitException e) {
            if (e instanceof PulsarClientException.ConsumerBusyException) {
                log.warn(((PulsarClientException.ConsumerBusyException) e).getLocalizedMessage());
                return null;
            }
            throw new ConsumerInitException("Failed to init consumer.", e);
        }
    }

}

@Component
@Slf4j
@Aspect
@EnableAspectJAutoProxy(proxyTargetClass = true, exposeProxy = true)
class PulsarProducerAspect {

    @Autowired
    private PulsarProducerTemplate pulsarProducerTemplate;

    @Pointcut("execution(* *..*(..)) && @annotation(cn.callmee.springboot.pulsar.starter.client.annotations.PulsarProducer)")
    public void processProducer() {
    }

    @Around("processProducer()")
    public Object afterProcessProducer(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        Object proceedResult = proceedingJoinPoint.proceed(proceedingJoinPoint.getArgs());
        // 对定义的生产者返回值进行非空判断
        if (null != proceedResult) {
            // 非空返回值则执行消息发送
            MethodSignature signature = (MethodSignature) proceedingJoinPoint.getSignature();
            Class returnType = signature.getReturnType();
            Method method = signature.getMethod();
            String name = method.getName();
            PulsarProducer annotationPulsarProducer = method.getAnnotation(PulsarProducer.class);
            String[] annotationTopicArr = SchemaUtils.generateProducerTopicList(annotationPulsarProducer);
            for (String annotationTopic : annotationTopicArr) {
                String topic = PulsarClientInitialConfiguration.STRING_VALUE_RESOLVER.resolveStringValue(StringUtils.hasText(annotationTopic) ? annotationTopic : name);
                boolean async = annotationPulsarProducer.async();
                if (async) {
                    pulsarProducerTemplate.sendAsync(topic, returnType.cast(proceedResult));
                } else {
                    pulsarProducerTemplate.send(topic, returnType.cast(proceedResult));
                }
                log.info("处理消息生产者: send message to {}: {}", topic, proceedResult);

            }
        }
        return proceedResult;
    }
}

