package com.told.sdk

import android.app.Application
import android.content.Context
import android.util.Log
import com.told.sdk.data.db.LocalStoreManager
import com.told.sdk.data.db.LocalStoreManagerImpl
import com.told.sdk.domain.GetTriggerableSurveysInteractor
import com.told.sdk.utils.Constants
import com.told.sdk.utils.InteractorWithParams
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext

object ToldSDK {
    private lateinit var getTriggerableSurveysInteractor: InteractorWithParams<GetTriggerableSurveysInteractor.Params, Result<GetEverySurveyAvailableToBeTriggeredQuery.Data>>
    private var sourceId: String? = null
    private lateinit var localStoreManager: LocalStoreManager
    private lateinit var screenTrigger: ScreenTrigger
    private lateinit var customEventTrigger: EventTrigger

    fun init(applicationContext: Context, sourceId: String ) {
        this.sourceId = sourceId
        localStoreManager = LocalStoreManagerImpl(applicationContext)
        getTriggerableSurveysInteractor = GetTriggerableSurveysInteractor(Constants.PRODUCTION_SERVER_API,)

        runBlocking {
            withContext(Dispatchers.IO) {
                getTriggerableSurveysInteractor.execute(
                    GetTriggerableSurveysInteractor.Params(
                        sourceId = sourceId,
                        applicationId = applicationContext.packageName,
                        applicationVersion = applicationContext.packageManager.getPackageInfo(
                            applicationContext.packageName,
                            0
                        ).versionName,
                        localStorageManagerInstance = localStoreManager //To be moved away with the dependency injection
                    )
                ).onSuccess { surveys ->
                    val surveysWeCanTrigger = surveys.getEverySurveyAvailableToBeTriggered ?: return@withContext
                    surveysWeCanTrigger.filter { it?.kind == "SurveyTriggerScreen" }?.let {
                        screenTrigger = ScreenTrigger(
                            application = applicationContext as Application,
                            sourceId = sourceId,
                            surveys = it,
                            widgetRootUrl = Constants.PRODUCTION_WIDGET_URL,
                            localStorageManagerInstance = localStoreManager
                        )
                    }

                    surveysWeCanTrigger.filter { it?.kind == "SurveyTriggerCustomEvent"}?.let {
                        customEventTrigger = EventTrigger(
                            sourceId = sourceId,
                            surveys = it,
                            widgetRootUrl = Constants.PRODUCTION_WIDGET_URL,
                            localStorageManagerInstance = localStoreManager
                        )
                    }
                }.onFailure {
                    Log.e("ToldSDK.init", it.message.orEmpty())
                }
            }
        }
    }

    fun trackEvent(
        context: Context,
        event: String,
        properties: List<Map<String, String>>
    ) {
        if (this::customEventTrigger.isInitialized)
            customEventTrigger.trackEvent(context = context, event = event, properties = properties)
    }

    fun addHiddenFields(
        data: List<Map<String, String>>
    ) {
        data?.let {
            if (this::screenTrigger.isInitialized)
                screenTrigger.udpdateHiddenFields(data= it)
            if (this::customEventTrigger.isInitialized)
                customEventTrigger.udpdateHiddenFields(data= it)
        }
    }
}
