package com.told.sdk

import android.app.Activity
import android.app.Application
import android.app.Application.ActivityLifecycleCallbacks
import android.content.Context
import android.os.Bundle
import android.util.Log
import com.told.sdk.data.db.LocalStoreManager
import com.told.sdk.domain.GetTriggerToDisplayInteractor
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext


internal abstract class ToldTrigger(
    internal val  surveys: List<GetEverySurveyAvailableToBeTriggeredQuery.GetEverySurveyAvailableToBeTriggered?>,
    internal val sourceId: String,
    internal val widgetRootUrl: String
) {

    private var widget: Widget? = null
    private var hiddenFields: String? = null

    protected fun launchWidget(
        context: Context,
        url: String,
        surveyId: String? = null
    ) {
        if (widget != null)
            return
        surveyId?.let {
            widget = Widget(
                context = context,
                url = url,
                surveyId = it,
                sourceId = sourceId,
                close = { setWidgetToNull() }
            )
        }
    }

    protected fun setWidgetToNull(){
        widget = null
    }
    protected fun closeWidget() {
        widget?.shutDownWebview()
    }
    protected fun buildWidgetUrl(surveyId: String?, sourceId: String): String {
        var url = "$widgetRootUrl?id=$surveyId&toldProjectID=$sourceId"
        if (hiddenFields != null)
            url += "&$hiddenFields"
        return url
    }

    private fun convertToQueryString(fields: List<Map<String, String>>): String {
        val queryString = StringBuilder()

        for (fieldMap in fields) {
            for ((key, value) in fieldMap) {
                if (queryString.isNotEmpty()) {
                    queryString.append("&")
                }
                queryString.append("$key=$value")
            }
        }

        return queryString.toString()
    }

    fun udpdateHiddenFields(data: List<Map<String, String>>) {
        hiddenFields = convertToQueryString(fields = data)
    }
}

internal class EventTrigger(
    sourceId: String,
    widgetRootUrl: String,
    surveys: List<GetEverySurveyAvailableToBeTriggeredQuery.GetEverySurveyAvailableToBeTriggered?>,
    localStorageManagerInstance: LocalStoreManager,
): ToldTrigger(surveys = surveys, sourceId = sourceId, widgetRootUrl = widgetRootUrl) {

    private val getTriggerToDisplayInteractor =
        GetTriggerToDisplayInteractor(localStorageManagerInstance)
    private var scope: CoroutineScope? = null

    internal fun trackEvent(
        context: Context,
        event: String,
        properties: List<Map<String, String>>
    ) {
        scope?.cancel()
        scope = CoroutineScope(Dispatchers.IO)
        scope?.launch {
            val surveyToDisplay = getTriggerToDisplayInteractor.execute(
                GetTriggerToDisplayInteractor.Params(
                    surveys = surveys,
                    activityName = null,
                    eventName = event,
                    eventProperties = properties,
                )
            )

            surveyToDisplay?.let {
                withContext(Dispatchers.Main) {
                    launchWidget(
                        context = context,
                        url = buildWidgetUrl(surveyId = it.survey, sourceId = sourceId),
                        surveyId = it.survey
                    )
                }
            }
        }
    }
}

internal class ScreenTrigger(
    application: Application,
    sourceId: String,
    widgetRootUrl: String,
    surveys: List<GetEverySurveyAvailableToBeTriggeredQuery.GetEverySurveyAvailableToBeTriggered?>,
    localStorageManagerInstance: LocalStoreManager,
) : ToldTrigger(surveys = surveys, sourceId = sourceId, widgetRootUrl = widgetRootUrl) {

    private val getTriggerToDisplayInteractor =
        GetTriggerToDisplayInteractor(localStorageManagerInstance)
    private var scope: CoroutineScope? = null

    init {
        observeLifecycle(
            application = application,
            surveys = surveys.filter { it?.kind == "SurveyTriggerScreen" })
    }

    private fun observeLifecycle(
        application: Application,
        surveys: List<GetEverySurveyAvailableToBeTriggeredQuery.GetEverySurveyAvailableToBeTriggered?>,
    ) {
        application.registerActivityLifecycleCallbacks(object : ActivityLifecycleCallbacks {
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
            }

            override fun onActivityStarted(activity: Activity) {
                scope?.cancel()
                scope = CoroutineScope(Dispatchers.IO)
                scope?.launch {
                    val surveyToDisplay = getTriggerToDisplayInteractor.execute(
                        GetTriggerToDisplayInteractor.Params(
                            surveys = surveys,
                            activityName = activity.javaClass.simpleName,
                            eventName = null,
                            eventProperties = null
                        )
                    )

                    surveyToDisplay?.let {
                        withContext(Dispatchers.Main) {
                            it.onSurveyTriggerScreen?.delay?.let { delay ->
                                delay.active.let { delay(
                                    (delay.value * 1000).toLong()
                                ) }
                            }
                            launchWidget(
                                context = activity,
                                url = buildWidgetUrl(surveyId = it.survey, sourceId = sourceId),
                                surveyId = it.survey
                            )
                        }
                    }
                }
            }

            override fun onActivityResumed(activity: Activity) {
            }

            override fun onActivityPaused(activity: Activity) {
                closeWidget()
                setWidgetToNull()
            }

            override fun onActivityStopped(activity: Activity) {

            }

            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {

            }

            override fun onActivityDestroyed(activity: Activity) {

            }

        })

    }
}