package com.told.sdk

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.net.Uri
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.webkit.JavascriptInterface
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.FrameLayout
import com.told.sdk.data.db.LocalStoreManager
import com.told.sdk.data.db.LocalStoreManagerImpl
import com.told.sdk.utils.Constants
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import org.json.JSONException
import org.json.JSONObject
import java.util.Date


class Widget(private val context: Context, private val url: String, private val surveyId: String, private val sourceId: String?, private val close: () -> Unit) {
    private val webView: WebView = WebView(context)
    private val localStoreManager: LocalStoreManager = LocalStoreManagerImpl(context)
    private var height: Int = 320
    init {
        createWebview()
    }

    private fun createWebview() {
        val webSettings: WebSettings = webView.settings
        webSettings.javaScriptEnabled = true
        webSettings.cacheMode = WebSettings.LOAD_NO_CACHE
        webSettings.domStorageEnabled = true;

        webView.addJavascriptInterface(WebAppInterface(context), "Android")

        webView.loadUrl(this.url)

        val layoutParams = FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            320
        )

        layoutParams.gravity = Gravity.BOTTOM
        webView.setBackgroundColor(Color.TRANSPARENT)

        // Ajoutez un WebViewClient pour attendre que la page soit chargée avant d'appliquer la hauteur maximale
        webView.webViewClient = object : WebViewClient() {
            override fun onPageFinished(view: WebView?, url: String?) {
                super.onPageFinished(view, url)
                webView.loadUrl(
                    "javascript:(function() {" +
                            "window.parent.addEventListener ('message', function(event) {" +
                            " Android.receiveMessage(JSON.stringify(event.data));});" +
                            "})()"
                )
                webView.loadUrl("javascript:window.postMessage({type: 'DEVICE_TYPE', value: 'phone'})")
                webView.layoutParams = layoutParams
            }
        }

        val rootView = (context as? Activity)?.window?.decorView?.findViewById<ViewGroup>(android.R.id.content)

        rootView?.let {
            it.addView(webView)
            webView.visibility = View.INVISIBLE
            webView.bringToFront()
        }
    }

    private fun showWebview() {
        webView.visibility = View.VISIBLE
    }

    internal fun shutDownWebview() {
        (context as? Activity)?.runOnUiThread {
            val rootView = (context as? Activity)?.window?.decorView?.findViewById<ViewGroup>(android.R.id.content)
            if (rootView == null)
                webView.visibility = View.GONE
            else
                (rootView as? ViewGroup)?.removeView(webView)
        }
        close()
    }

    private fun updateHeight(newHeight: Int) {
        (context as? Activity)?.runOnUiThread {
            val layoutParams = webView.layoutParams
            layoutParams.height = (newHeight * context?.resources?.displayMetrics?.density!!).toInt()
            webView.layoutParams = layoutParams
            this.height = newHeight
        }
    }


    private suspend fun addCookie(context: Context, surveyId: String, sourceId: String, reply: Boolean) {
        val surveyList = localStoreManager.getSurveyList(key = Constants.TOLD_STORE).firstOrNull()?.toMutableList() ?: mutableListOf()

        val existingSurvey = surveyList.find { it.surveyId == surveyId }
        if (existingSurvey != null) {
            existingSurvey.date = Date().toString()
            existingSurvey.reply = reply
        } else {
            val newSurvey = LocalStoreManager.Survey(surveyId, sourceId, Date().toString(), reply)
            surveyList.add(newSurvey)
        }

        Log.d("ToldSDK.addCookie", "surveyList => $surveyList")
        localStoreManager.setSurveyList(key = Constants.TOLD_STORE, surveyList = surveyList)
    }

    private inner class WebAppInterface(private val mContext: Context) {
        @JavascriptInterface
        fun receiveMessage(message: String) {
            try {
                GlobalScope.launch(Dispatchers.Main) {
                    val jsonMessage = JSONObject(message)
                    val messageType = jsonMessage.getString("type")

                    when (messageType) {
                        "ADD_COOKIE" -> {
                            addCookie(context = context, jsonMessage.getString("id"), jsonMessage.getString("projectId"), jsonMessage.getBoolean("reply"))
                        }

                        "IS_LOADED" -> {
                            if (jsonMessage.getString("id") == surveyId)
                                showWebview()
                        }

                        "HEIGHT_CHANGE" -> {
                            if (jsonMessage.getString("id") == surveyId) {
                                updateHeight(jsonMessage.getInt("value"));
                            }
                        }

                        "CLOSE" -> {
                            if (jsonMessage.getString("id") == surveyId)
                                shutDownWebview()
                        }

                        "CLOSE_CALENDAR" -> {
                        }

                        "LAUNCH_CALENDAR" -> {
                            jsonMessage.getJSONObject("value").getString("iframeUrl").let{iframeUrl ->
                                val browserIntent =
                                    Intent(Intent.ACTION_VIEW, Uri.parse(iframeUrl))
                                val activity = context as? Activity
                                activity?.startActivity(browserIntent)
                            }
                        }
                    }
                }
            } catch (e: JSONException) {
                Log.e("WebAppInterface", "Error parsing JSON message: $message")
            }
        }
    }
}