package com.told.sdk

import android.content.Context
import androidx.navigation.NavController
import kotlinx.coroutines.Dispatchers
import kotlin.coroutines.CoroutineContext
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds

/**
 * [Told] object is the entry point to interact with the SDK.
 */
object Told {
    private val toldDelegate: ToldDelegate by lazy { ToldDi.toldDelegate }
    private val toldActivityLifecycleCallback: ToldActivityLifecycleCallback by lazy { ToldDi.toldActivityLifecycleCallback }

    /**
     * This method MUST be called before any other interactions with the SDK; otherwise, unexpected behaviors may occur.
     * Usually, you should call this method from your custom [android.app.Application].
     * ```
     * class MyApplication : Application() {
     *     override fun onCreate() {
     *         super.onCreate()
     *         Told.init(
     *             configuration = ToldConfiguration(
     *                 sourceId = "MySourceId",
     *                 applicationId = applicationContext.packageName,
     *                 environment = ToldEnvironment.Production,
     *             ),
     *             applicationContext = applicationContext,
     *         )
     *     }
     * }
     * ```
     * @params configuration - See [ToldConfiguration].
     * @params applicationContext - Use the one from your application.
     * @params coroutineContext - Provide the desired [Dispatchers]. All requests made by Told will be executed on this [Dispatchers].
     * @return - true if we managed to verify the app and fetch the [AnonymousId], false otherwise.
     */
    suspend fun init(
        configuration: ToldConfiguration,
        applicationContext: Context,
        coroutineContext: CoroutineContext = Dispatchers.Default,
    ): Boolean {
        if (ToldDi.isAlreadyInitialized) {
            ToldLogger.i(message = "Reloading Told Di init")
            ToldDi.reload(configuration.environment)
        } else {
            ToldDi.start(
                applicationContext = applicationContext,
                coroutineContext = coroutineContext,
                toldEnvironment = configuration.environment,
            )
        }
        return toldDelegate.launch(configuration = configuration)
    }

    /**
     * Registers a navigation controller to listen for destination changes and perform logging and tracking actions.
     * This function adds an `OnDestinationChangedListener` to the provided `navController`. Each time the navigation
     * destination changes, an event is send to Told.
     * The `destinationName` is the full qualified name of your destination. If you have a destination called `MyDestination`,
     * the name send will be my.package.MyDestination. This name must match the name you enter on Told website.
     * @param navController - The [NavController] instance to monitor for navigation destination changes. Works with Compose or Fragment.
     */
    fun registerNavController(
        navController: NavController,
    ) {
        navController.addOnDestinationChangedListener { _, destination, _ ->
            ToldLogger.d(message = "Navigating to destination ${destination.route}")
            toldDelegate.trackPageChanged(destinationName = destination.route.orEmpty())
        }
    }

    /**
     * Allows listening to activity changes throughout the application and launching a survey if needed.
     * This method should be called only once during your application’s lifecycle.
     * If your application’s navigation is entirely based on one or more NavController, you should use only [registerNavController].
     * If your navigation is activity-based, you can use [registerActivityCallback].
     */
    fun registerActivityCallback() {
        toldActivityLifecycleCallback.init()
    }

    /**
     * If you need to unregister your callback manually, you can call this method.
     * If you use [registerActivityCallback] in your [android.app.Application] or if you only use [registerNavController], this method
     * should be useless for you.
     */
    fun unregisterActivityCallback() {
        toldActivityLifecycleCallback.unregister()
    }

    /**
     * Track any custom event you declare on Told.
     * @param eventName - Name declared on the website.
     * @param properties - Optional additional properties associated to your custom event.
     */
    fun trackEvent(
        eventName: String,
        properties: Map<String, Any> = emptyMap(),
    ) {
        toldDelegate.trackEvent(
            eventName = eventName,
            properties = properties,
        )
    }

    /**
     * Start a survey with the provided [surveyId].
     * @param surveyId - Id of the survey to start.
     * @param withDelay - Optional, time to wait before starting the survey (default to 0 second).
     */
    fun start(surveyId: String, withDelay: Duration = 0.seconds) {
        toldDelegate.start(surveyId = surveyId, withDelay = withDelay)
    }

    /**
     * Add custom properties to the current user.
     * @param properties - Map of yours properties.
     */
    fun identify(properties: Map<String, Any>) {
        toldDelegate.identify(properties = properties)
    }

    /**
     * Get information about your source and your surveys.
     * It will be displayed in your Logcat as a debug log.
     */
    fun debugWidget() {
        toldDelegate.debugWidget()
    }

    /**
     * Reset all data linked to your current SDK session.
     * No need to call [Told.init] again! The SDK will still be working.
     * @param newConfiguration - Provide a new configuration if you want to reset the SDK and change some data (environment for example).
     * If not provided, the same configuration will be used, but data (like anonymousId, will be reset).
     * @return - true if we managed to change the [AnonymousId], false otherwise.
     */
    suspend fun reset(newConfiguration: ToldConfiguration? = null): Boolean {
        return toldDelegate.reset(newConfiguration = newConfiguration)
    }
}
