package com.told.sdk

import android.app.Activity
import android.app.Application
import android.app.Application.ActivityLifecycleCallbacks
import android.content.Context
import android.os.Bundle

/**
 * A utility object to handle activity lifecycle callbacks for logging and tracking purposes.
 * This object provides functionality to monitor activity lifecycle events throughout the application.
 */
internal class ToldActivityLifecycleCallback(
    private val applicationContext: Context,
    private val toldDelegate: ToldDelegate,
) {
    private var callback: ActivityLifecycleCallbacks? = null

    /**
     * Initializes the lifecycle callback and registers it with the application.
     * This function creates and registers an `ActivityLifecycleCallbacks` instance to monitor activity lifecycle events.
     */
    fun init() {
        val callback = object : ActivityLifecycleCallbacks {
            override fun onActivityResumed(activity: Activity) {
                ToldLogger.d(message = "Activity ${activity::class.qualifiedName} displayed.")
                toldDelegate.trackPageChanged(destinationName = activity::class.qualifiedName.orEmpty())
            }

            override fun onActivityPaused(activity: Activity) {
                ToldLogger.d(message = "Cancelling display for activity ${activity::class.qualifiedName}.")
                toldDelegate.cancelPreviousSurvey()
            }

            override fun onActivityStopped(activity: Activity) = Unit
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) = Unit
            override fun onActivityStarted(activity: Activity) = Unit
            override fun onActivitySaveInstanceState(activity: Activity, savedInstanceState: Bundle) = Unit
            override fun onActivityDestroyed(activity: Activity) = Unit
        }
        // safe guard in case init() is called multiple without unregister() in between
        unregister()
        this.callback = callback
        (applicationContext as? Application)?.registerActivityLifecycleCallbacks(callback)
            ?: ToldLogger.e(message = "Context provided is not an application context")
    }

    fun unregister() {
        callback?.let { (applicationContext as? Application)?.unregisterActivityLifecycleCallbacks(it) }
        callback = null
    }
}
