package com.told.sdk

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.Gravity
import android.webkit.WebSettings
import android.webkit.WebView
import android.widget.FrameLayout
import androidx.activity.ComponentActivity
import androidx.activity.SystemBarStyle
import androidx.activity.compose.BackHandler
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.EnterTransition
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.navigationBarsPadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.statusBarsPadding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import org.koin.androidx.compose.koinViewModel
import org.koin.compose.KoinIsolatedContext
import android.graphics.Color as LegacyColor

internal class ToldWidgetActivity : ComponentActivity() {

    @SuppressLint("SetJavaScriptEnabled")
    @Suppress("LongMethod")
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge(
            statusBarStyle = SystemBarStyle.auto(LegacyColor.TRANSPARENT, LegacyColor.TRANSPARENT),
            navigationBarStyle = SystemBarStyle.light(LegacyColor.TRANSPARENT, LegacyColor.TRANSPARENT),
        )

        setContent {
            KoinIsolatedContext(context = ToldDiIsolatedContext.getSafeKoinApp()) {
                val viewModel: ToldWidgetViewModel = koinViewModel()
                val webViewState: ToldWebViewState by viewModel.webViewState.collectAsStateWithLifecycle()
                var isLoadingVisible: Boolean by rememberSaveable { mutableStateOf(true) }
                BackHandler {
                    viewModel.trackClose()
                    finish()
                }
                Box(
                    modifier = Modifier
                        .fillMaxSize()
                        .padding(all = 16.dp)
                        .statusBarsPadding()
                        .navigationBarsPadding()
                        .imePadding(),
                ) {
                    AnimatedVisibility(
                        visible = isLoadingVisible,
                        enter = EnterTransition.None,
                        exit = fadeOut(),
                        modifier = Modifier
                            .align(alignment = Alignment.BottomCenter),
                    ) {
                        Box(
                            modifier = Modifier
                                .fillMaxWidth()
                                .height(height = 200.dp)
                                .clip(RoundedCornerShape(32.dp))
                                .background(color = Color(0xFF1F2469)), // TODO <ui> see with Told team.
                            contentAlignment = Alignment.Center,
                        ) {
                            CircularProgressIndicator(
                                color = Color.White,
                            )
                            Image(
                                painter = painterResource(R.drawable.told), // TODO <ui> see with Told team.
                                contentDescription = null,
                                modifier = Modifier
                                    .align(Alignment.BottomStart)
                                    .padding(all = 24.dp)
                                    .width(width = 38.dp),
                            )
                        }
                    }
                    when (val safeState = webViewState) {
                        is ToldWebViewState.Loading -> Unit
                        is ToldWebViewState.Failure -> {
                            ToldLogger.e(message = "An error occurred when trying to display the WebView.")
                            finish()
                        }
                        is ToldWebViewState.Success -> {
                            AndroidView(
                                factory = { context ->
                                    val webView = WebView(context)
                                    val webSettings: WebSettings = webView.settings
                                    webSettings.javaScriptEnabled = true
                                    webSettings.cacheMode = WebSettings.LOAD_NO_CACHE
                                    webSettings.domStorageEnabled = true
                                    webView.loadUrl(safeState.url)
                                    val layoutParams = FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, 200)
                                    layoutParams.gravity = Gravity.BOTTOM
                                    webView.setBackgroundColor(LegacyColor.TRANSPARENT)
                                    webView.addJavascriptInterface(
                                        ToldWebViewInterface(
                                            surveyId = viewModel.surveyId,
                                            webView = webView,
                                            activity = this@ToldWidgetActivity,
                                            onWebViewReady = { if (isLoadingVisible) isLoadingVisible = false },
                                            trackClose = viewModel::trackClose,
                                        ),
                                        "Android",
                                    )
                                    webView.webViewClient = ToldWebViewClient(webView = webView)
                                    webView
                                },
                                modifier = Modifier
                                    .align(alignment = Alignment.BottomCenter)
                                    .clip(RoundedCornerShape(32.dp)),
                            )
                        }
                    }
                }
            }
        }
    }

    companion object {
        internal const val SurveyId: String = "SurveyId"
        fun start(surveyId: String, context: Context) {
            context.startActivity(
                Intent(
                    context,
                    ToldWidgetActivity::class.java,
                ).apply {
                    flags = Intent.FLAG_ACTIVITY_NEW_TASK
                    putExtra(SurveyId, surveyId)
                },
            )
        }
    }
}
