package com.told.sdk

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.told.sdk.ToldWidgetActivity.Companion.SurveyId
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch

internal class ToldWidgetViewModel(
    savedStateHandle: SavedStateHandle,
    private val toldDelegate: ToldDelegate,
) : ViewModel() {
    val surveyId: SurveyId? = savedStateHandle.get<String?>(SurveyId)?.let { SurveyId(value = it) }

    private val _webViewState: MutableStateFlow<ToldWebViewState> = MutableStateFlow(ToldWebViewState.Loading)
    val webViewState: StateFlow<ToldWebViewState> = _webViewState.asStateFlow()

    init {
        viewModelScope.launch {
            val sourceId = toldDelegate.getSourceId()
            val anonymousId = toldDelegate.getAnonymousId()
            val environment = toldDelegate.getToldEnvironment()
            @Suppress("ComplexCondition")
            if (sourceId == null || anonymousId == null || environment == null || surveyId == null) {
                ToldLogger.e(message = "Missing value: sourceId, anonymousId and/or environment. Did you call Told.init?")
                _webViewState.value = ToldWebViewState.Failure
            } else {
                val url = "${environment.widgetUrl}?id=${surveyId.value}&sourceID=${sourceId.value}&anonymousID=${anonymousId.value}"
                _webViewState.value = ToldWebViewState.Success(url = url)
                ToldLogger.d(message = "Prepare displaying in WebView with URL: $url")
            }
        }
    }

    fun trackClose() {
        viewModelScope.launch {
            toldDelegate.trackCloseEvent(surveyId = surveyId)
        }
    }
}
