package com.told.sdk

import android.content.Context
import com.told.sdk.LogUtil.logLongDebug
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlin.coroutines.CoroutineContext
import kotlin.time.Duration

internal class ToldDelegate(
    coroutineContext: CoroutineContext,
    private val applicationContext: Context,
    private val toldQueries: ToldQueries,
    private val toldMutations: ToldMutations,
    private val toldDatastore: ToldDatastore,
) {
    private val coroutineScope = CoroutineScope(context = coroutineContext)

    private var startJob: Job? = null

    suspend fun launch(
        configuration: ToldConfiguration,
    ): Boolean {
        return if (checkIfAppIsAllowed(configuration = configuration)) {
            updateLanguage(language = configuration.language)
            toldDatastore.store(value = configuration.applicationId, key = ToldDatastore.ApplicationIdKey)
            toldDatastore.store(value = configuration.appVersion, key = ToldDatastore.AppVersionKey)
            toldDatastore.store(value = configuration.sourceId, key = ToldDatastore.SourceIdKey)
            toldDatastore.store(value = configuration.environment.name.name, key = ToldDatastore.EnvironmentNameKey)
            toldDatastore.store(value = configuration.environment.serverUrl, key = ToldDatastore.EnvironmentServerUrlKey)
            toldDatastore.store(value = configuration.environment.widgetUrl, key = ToldDatastore.EnvironmentWidgetUrlKey)
            if (verifyAnonymousIdIsAvailable()) {
                true
            } else {
                toldDatastore.clearDatastore()
                false
            }
        } else {
            false
        }
    }

    fun updateLanguage(language: String? = null) {
        coroutineScope.launch {
            val currentLocale = language ?: applicationContext.resources.configuration.locales[0]?.language
            toldDatastore.store(value = currentLocale.orEmpty(), key = ToldDatastore.LanguageKey)
        }
    }

    fun trackEvent(
        eventName: String,
        properties: Map<String, Any> = emptyMap(),
    ) {
        coroutineScope.launch {
            val anonymousId: AnonymousId? = getAnonymousId()
            val sourceId: SourceId? = getSourceId()
            if (anonymousId != null && sourceId != null) {
                ToldLogger.d(message = "Track event: $eventName with ${properties.size} properties.")
                toldMutations.trackCustomEvent(
                    anonymousId = anonymousId,
                    sourceId = sourceId,
                    customName = eventName,
                    customData = properties,
                    primaryData = getToldPrimaryData(),
                ).getOrNull()?.let { result ->
                    ToldLogger.d(message = "Successfully track event $eventName.")
                    when (result) {
                        is TrackEventResult.NoTrigger -> ToldLogger.d(message = "Condition not filled for event $eventName.")
                        is TrackEventResult.TriggerOn -> {
                            ToldLogger.d(message = "Condition filled for event $eventName. Trying to launch survey.")
                            Told.start(surveyId = result.surveyId.value)
                        }
                    }
                } ?: ToldLogger.e(message = "Failed to track event: $eventName")
            } else {
                ToldLogger.d(message = "Can't track event because anonymousId or sourceId is null.")
            }
        }
    }

    fun trackPageChanged(
        destinationName: String,
    ) {
        coroutineScope.launch {
            cancelPreviousSurvey()
            val anonymousId: AnonymousId? = getAnonymousId()
            val sourceId: SourceId? = getSourceId()
            if (sourceId != null && anonymousId != null) {
                val previousScreenDisplayed = toldDatastore.get(ToldDatastore.LastPageDisplayedKey)
                toldDatastore.store(value = destinationName, key = ToldDatastore.LastPageDisplayedKey)
                // Check mandatory when we handle activities. Avoid infinite loop tracking.
                val toldWidgetActivity = ToldWidgetActivity::class.qualifiedName
                if (previousScreenDisplayed != toldWidgetActivity && destinationName != toldWidgetActivity) {
                    val trackResult = toldMutations.trackPageChangedEvent(
                        anonymousId = anonymousId,
                        sourceId = sourceId,
                        primaryData = getToldPrimaryData(),
                    )
                    trackResult.getOrNull()?.let { result ->
                        when (result) {
                            is TrackEventResult.NoTrigger -> ToldLogger.d(message = "No survey to display for route $destinationName")
                            is TrackEventResult.TriggerOn -> {
                                ToldLogger.d(message = "Trying to display survey for route $destinationName")
                                Told.start(surveyId = result.surveyId.value, withDelay = result.withDelay)
                            }
                        }
                    } ?: ToldLogger.e(message = "Failed to track page changed.")
                }
            } else {
                ToldLogger.d(message = "Can't track page changed because anonymousId or sourceId is null.")
            }
        }
    }

    fun start(
        surveyId: String,
        withDelay: Duration,
    ) {
        startJob?.cancel()
        startJob = coroutineScope.launch {
            val environment: ToldEnvironment? = getToldEnvironment()
            val applicationId: String? = getApplicationId()
            if (applicationId != null && environment != null) {
                val checkResult = toldQueries.checkIfCanUseWidgetWithSurvey(
                    surveyId = SurveyId(value = surveyId),
                    applicationId = applicationId,
                    environment = environment,
                )
                checkResult.getOrNull()?.let { canUse ->
                    if (canUse) {
                        ToldLogger.d(message = "Delay display for $withDelay")
                        delay(withDelay)
                        ToldWidgetActivity.start(
                            surveyId = surveyId,
                            context = applicationContext,
                        )
                    } else {
                        ToldLogger.d(message = "Survey with id $surveyId can not be displayed.")
                    }
                }
            } else {
                ToldLogger.d(message = "Can't start because applicationId or environment is null.")
            }
        }
    }

    fun debugWidget() {
        coroutineScope.launch {
            val environment: ToldEnvironment? = getToldEnvironment()
            val sourceId: SourceId? = getSourceId()
            if (environment != null && sourceId != null) {
                toldQueries.debugWidget(sourceId = sourceId, environment = environment).getOrNull()?.logLongDebug()
            }
        }
    }

    fun identify(properties: Map<String, Any>) {
        coroutineScope.launch {
            val anonymousId: AnonymousId? = getAnonymousId()
            val sourceId: SourceId? = getSourceId()
            if (anonymousId != null && sourceId != null) {
                val result = toldMutations.identify(
                    anonymousId = anonymousId,
                    sourceId = sourceId,
                    properties = properties,
                    primaryData = getToldPrimaryData(),
                )
                if (result.isSuccess) {
                    result.getOrNull()?.let { newAnonymousId ->
                        ToldLogger.d(message = "Properties $properties has been saved and anonymous ID is now ${newAnonymousId.value}")
                        toldDatastore.store(value = newAnonymousId.value, key = ToldDatastore.AnonymousIdKey)
                    }
                } else {
                    ToldLogger.d(message = "Something went wrong when trying to save properties.", throwable = result.exceptionOrNull())
                }
            } else {
                ToldLogger.d(message = "Can't identify because anonymousId or sourceId is null.")
            }
        }
    }

    suspend fun trackCloseEvent(surveyId: SurveyId?) {
        val anonymousId: AnonymousId? = getAnonymousId()
        val sourceId: SourceId? = getSourceId()
        if (sourceId != null && anonymousId != null) {
            toldMutations.trackCloseEvent(
                anonymousId = anonymousId,
                sourceId = sourceId,
                primaryData = getToldPrimaryData(surveyId = surveyId),
            )
        } else {
            ToldLogger.d(message = "Can't track close event because anonymousId or sourceId is null.")
        }
    }

    fun cancelPreviousSurvey() {
        startJob?.cancel()
        startJob = null
    }

    suspend fun getAnonymousId(): AnonymousId? {
        val anonymousId: AnonymousId? = toldDatastore.get(key = ToldDatastore.AnonymousIdKey)?.let { AnonymousId(it) }
        if (anonymousId == null) ToldLogger.e(message = "Missing value for anonymousId. Did you called Told.init?")
        return anonymousId
    }

    suspend fun getSourceId(): SourceId? {
        val sourceId: SourceId? = toldDatastore.get(key = ToldDatastore.SourceIdKey)?.let { SourceId(it) }
        if (sourceId == null) ToldLogger.e(message = "Missing value for sourceId. Did you called Told.init?")
        return sourceId
    }

    suspend fun getToldEnvironment(): ToldEnvironment? {
        val environmentName: ToldEnvironment.EnvironmentName? =
            toldDatastore.get(key = ToldDatastore.EnvironmentNameKey)?.let(ToldEnvironment.EnvironmentName::valueOf)
        val serverUrl: String? = toldDatastore.get(key = ToldDatastore.EnvironmentServerUrlKey)
        val widgetUrl: String? = toldDatastore.get(key = ToldDatastore.EnvironmentWidgetUrlKey)
        if (environmentName == null || serverUrl == null || widgetUrl == null) {
            ToldLogger.e(message = "Missing value for environment. Did you called Told.init?")
            return null
        }
        return when (environmentName) {
            ToldEnvironment.EnvironmentName.Production -> ToldEnvironment.Production
            ToldEnvironment.EnvironmentName.Development -> ToldEnvironment.Development
            ToldEnvironment.EnvironmentName.Custom -> ToldEnvironment.Custom(serverUrl = serverUrl, widgetUrl = widgetUrl)
        }
    }

    suspend fun getApplicationId(): String? {
        val applicationId: String? = toldDatastore.get(key = ToldDatastore.ApplicationIdKey)
        if (applicationId == null) ToldLogger.e(message = "Missing value for applicationId. Did you called Told.init?")
        return applicationId
    }

    private suspend fun getToldPrimaryData(surveyId: SurveyId? = null): ToldPrimaryData {
        return ToldPrimaryData(
            language = toldDatastore.get(ToldDatastore.LanguageKey),
            pageName = toldDatastore.get(ToldDatastore.LastPageDisplayedKey),
            surveyId = surveyId,
            appVersion = toldDatastore.get(ToldDatastore.AppVersionKey),
        )
    }

    /**
     * Check if app is allowed to be used, depending on provided [ToldConfiguration].
     * @param configuration - [ToldConfiguration] created by user.
     * @return - true if Told is configured correctly, false otherwise.
     */
    private suspend fun checkIfAppIsAllowed(configuration: ToldConfiguration): Boolean {
        val isAppAllowedResult = toldQueries.checkIfAppIsAllowed(
            sourceId = configuration.sourceId,
            applicationId = configuration.applicationId,
            environment = configuration.environment,
        )
        return isAppAllowedResult.getOrNull()?.let { isAppAllowed ->
            ToldLogger.d(message = "Your app is ${if (isAppAllowed) "allowed" else "not allowed"}.")
            isAppAllowed
        } ?: run {
            ToldLogger.d(message = "An error occurred during the verification of your app.")
            false
        }
    }

    /**
     * Verify if we already have a [AnonymousId] locally, otherwise fetch it from Told backend.
     * @return - true if we managed to fetch the [AnonymousId] or we already have one, false otherwise.
     */
    private suspend fun verifyAnonymousIdIsAvailable(): Boolean {
        val localAnonymousId = toldDatastore.get(key = ToldDatastore.AnonymousIdKey)
        return if (localAnonymousId == null) {
            toldMutations.getAnonymousId().getOrNull()?.let { newAnonymousId ->
                toldDatastore.store(value = newAnonymousId.value, key = ToldDatastore.AnonymousIdKey)
                ToldLogger.d(message = "anonymousId has been saved: $newAnonymousId")
                true
            } == true
        } else {
            ToldLogger.d(message = "anonymousId is $localAnonymousId (unchanged from last execution)")
            true
        }
    }

    suspend fun reset(newConfiguration: ToldConfiguration?): Boolean {
        val anonymousId = getAnonymousId()
        val sourceId = getSourceId()
        // could be null in case of a previous reset and if the new configuration is not allowed.
        if (anonymousId != null && sourceId != null) {
            toldMutations.trackResetEvent(
                anonymousId = anonymousId,
                sourceId = sourceId,
                primaryData = getToldPrimaryData(),
            )
        }
        return newConfiguration?.let {
            ToldLogger.d(message = "New configuration provided. Reset all previous data.")
            ToldDi.reset(toldEnvironment = newConfiguration.environment)
            launch(configuration = newConfiguration)
        } ?: run {
            toldDatastore.clear(key = ToldDatastore.AnonymousIdKey)
            if (verifyAnonymousIdIsAvailable()) {
                true
            } else {
                ToldLogger.d(message = "Reset failed, putting back previous anonymousId.")
                anonymousId?.value?.let { anonymousIdValue ->
                    toldDatastore.store(value = anonymousIdValue, key = ToldDatastore.AnonymousIdKey)
                }
                false
            }
        }
    }
}
