package com.told.sdk

import android.content.Context
import org.koin.android.ext.koin.androidContext
import org.koin.core.Koin
import org.koin.core.KoinApplication
import org.koin.core.component.KoinComponent
import org.koin.core.module.Module
import org.koin.core.module.dsl.singleOf
import org.koin.core.module.dsl.viewModelOf
import org.koin.dsl.koinApplication
import org.koin.dsl.module
import kotlin.coroutines.CoroutineContext

/**
 * Initialize all object that will be used internally by the SDK.
 * We use Koin for injection here.
 */
internal object ToldDi : ToldIsolatedKoinComponent {
    val toldDelegate: ToldDelegate
        get() = getKoin().get()

    val toldActivityLifecycleCallback: ToldActivityLifecycleCallback
        get() = getKoin().get()

    val isAlreadyInitialized: Boolean
        get() = ToldDiIsolatedContext.isAlreadyInitialized

    /**
     * Should be called directly before any other use of the SDK.
     * Remember that we can only have one instance of KoinApplication.
     * We load/unload modules dynamically to allow modification on data if user wants to.
     * Calling start multiples times will result in a new configuration built each time.
     */
    fun start(
        applicationContext: Context,
        toldEnvironment: ToldEnvironment,
        coroutineContext: CoroutineContext,
    ) {
        ToldDiIsolatedContext.start(
            applicationContext = applicationContext,
            toldEnvironment = toldEnvironment,
            coroutineContext = coroutineContext,
        )
        ToldLogger.i(message = "Told is ready to be used!")
    }

    suspend fun reset(toldEnvironment: ToldEnvironment) {
        getKoin().get<ToldDatastore>().clearDatastore()
        reload(toldEnvironment)
    }

    fun reload(toldEnvironment: ToldEnvironment) {
        ToldDiIsolatedContext.unload()
        ToldDiIsolatedContext.load(toldEnvironment = toldEnvironment)
    }
}

/**
 * Isolated context for Koin associated to our SDK.
 * User will be able to start their own koinApplication without interfering with the SDK.
 */
internal object ToldDiIsolatedContext {
    private var koinApp: KoinApplication? = null
    val koin: Koin
        get() = getSafeKoinApp().koin

    val isAlreadyInitialized: Boolean
        get() = koinApp != null

    /**
     * Environment module is dynamic as we can change environment during app session.
     * Others instances don't need to be refreshed.
     */
    private var environmentModule: Module? = null

    fun start(
        applicationContext: Context,
        toldEnvironment: ToldEnvironment,
        coroutineContext: CoroutineContext,
    ) {
        if (koinApp == null) {
            koinApp = koinApplication {
                androidContext(applicationContext)
                modules(
                    module {
                        single { get<Context>().filesDir.resolve(DatastoreName).absolutePath }
                        single { coroutineContext }
                        singleOf(::ToldActivityLifecycleCallback)
                        singleOf(::ToldDatastore)
                        singleOf(::ToldDelegate)
                        singleOf(::ToldMutations)
                        singleOf(::ToldQueries)
                        viewModelOf(::ToldWidgetViewModel)
                    },
                )
            }
            load(toldEnvironment = toldEnvironment)
        }
    }

    fun load(toldEnvironment: ToldEnvironment) {
        environmentModule = module {
            single { toldEnvironment }
            singleOf(::ToldClient)
        }
        koin.loadModules(listOfNotNull(environmentModule))
    }

    fun unload() {
        koin.unloadModules(listOfNotNull(environmentModule))
    }

    fun getSafeKoinApp(): KoinApplication {
        return checkNotNull(koinApp) { "Did you call start()?" }
    }

    private const val DatastoreName = "told.preferences_pb"
}

/**
 * Koin Component associated to our SDK.
 */
internal interface ToldIsolatedKoinComponent : KoinComponent {
    override fun getKoin(): Koin = ToldDiIsolatedContext.koin
}
