package com.told.sdk

import android.animation.ValueAnimator
import android.content.Intent
import android.net.Uri
import android.util.Log
import android.webkit.JavascriptInterface
import android.webkit.WebView
import androidx.activity.ComponentActivity
import androidx.core.animation.doOnEnd
import org.json.JSONException
import org.json.JSONObject
import kotlin.time.Duration.Companion.milliseconds

internal class ToldWebViewInterface(
    private val surveyId: SurveyId?,
    private val webView: WebView,
    private val activity: ComponentActivity,
    private val onWebViewReady: () -> Unit,
    private val trackClose: () -> Unit,
) {
    private var previousHeight: Int = 200

    @Suppress("unused")
    @JavascriptInterface
    fun receiveMessage(message: String) {
        activity.runOnUiThread {
            try {
                val jsonMessage = JSONObject(message)
                when (jsonMessage.getString("type")) {
                    ToldWidgetEvent.OpenLink.rawValue,
                    ToldWidgetEvent.ReplaceUrl.rawValue -> {
                        jsonMessage.getJSONObject("value").getString("url").let { url ->
                            val browserIntent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
                            activity.startActivity(browserIntent)
                        }
                    }
                    ToldWidgetEvent.IsLoaded.rawValue -> {
                        val id = jsonMessage.getString("id")
                        if (id == surveyId?.value) webView.loadUrl("javascript:window.postMessage({type: 'DEVICE_TYPE', value: 'phone'})")
                    }
                    ToldWidgetEvent.HeightChange.rawValue -> {
                        if (jsonMessage.getString("id") == surveyId?.value) {
                            val newHeight = jsonMessage.getInt("value")
                            val animator = ValueAnimator.ofInt(previousHeight, newHeight)
                            animator.addUpdateListener { valueAnimator ->
                                val animatedValue = valueAnimator.animatedValue as Int
                                val layoutParams = webView.layoutParams
                                layoutParams.height = (animatedValue * activity.resources?.displayMetrics?.density!!).toInt()
                                webView.layoutParams = layoutParams
                            }
                            animator.doOnEnd {
                                previousHeight = newHeight
                                onWebViewReady()
                            }
                            animator.duration = 250.milliseconds.inWholeMilliseconds
                            animator.start()
                        }
                    }
                    ToldWidgetEvent.Close.rawValue -> {
                        if (jsonMessage.getString("id") == surveyId?.value) {
                            trackClose()
                            activity.finish()
                        }
                    }
                    ToldWidgetEvent.LaunchCalendar.rawValue -> {
                        jsonMessage.getJSONObject("value").getString("iframeUrl").let { iframeUrl ->
                            val browserIntent = Intent(Intent.ACTION_VIEW, Uri.parse(iframeUrl))
                            activity.startActivity(browserIntent)
                        }
                    }
                    ToldWidgetEvent.CloseCalendar.rawValue -> {
                        // Not sure what to do with this for now
                    }
                }
            } catch (e: JSONException) {
                Log.e("WebAppInterface", "Error parsing JSON message: $message")
            }
        }
    }
}
