package com.told.sdk

object LogUtil {
    private const val MAX_LOG_LENGTH = 4000 // Android's log message limit is ~4kb

    /**
     * Splits a large message into chunks and logs each chunk with debug priority.
     * Adds continuation markers to show message sequence.
     *
     * @param message The message to log
     * @param logger Optional custom logging function. Defaults to ToldLogger.d
     */
    fun logLongDebug(
        message: String,
        logger: (String) -> Unit = { m -> ToldLogger.d(m) },
    ) {
        // If message is within limits, log it directly
        if (message.length <= MAX_LOG_LENGTH) {
            logger(message)
            return
        }

        // Split message into chunks
        var remainingMessage = message
        var chunkIndex = 0
        val totalChunks = kotlin.math.ceil(message.length.toDouble() / MAX_LOG_LENGTH).toInt()

        while (remainingMessage.isNotEmpty()) {
            chunkIndex++
            val chunkSize = kotlin.math.min(remainingMessage.length, MAX_LOG_LENGTH)
            val chunk = remainingMessage.substring(0, chunkSize)

            // Format: [1/3] Message chunk
            val chunkHeader = "[$chunkIndex/$totalChunks] "
            val logMessage = "$chunkHeader$chunk"
            logger(logMessage)

            // Remove logged chunk from remaining message
            remainingMessage = remainingMessage.substring(chunkSize)
        }
    }

    /**
     * Extension function to make logging more convenient
     */
    fun String.logLongDebug() = logLongDebug(this)
}
