package cn.huermao.framework.aspectJ;

import cn.huermao.framework.annotation.Log;
import cn.huermao.framework.base.BaseContextHandler;
import cn.huermao.framework.constant.Constants;
import cn.huermao.framework.domain.LogInfo;
import cn.huermao.framework.enums.HttpMethod;
import cn.huermao.framework.enums.Terminal;
import cn.huermao.framework.utils.JsonUtils;
import cn.huermao.framework.utils.ServletUtils;
import cn.huermao.framework.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.HandlerMapping;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;

/**
 * 操作日志记录处理
 *
 * @author Lion Li
 */
@Slf4j
@Aspect
@Component
public class LogAspect {

    /**
     * 处理完请求后执行
     *
     * @param joinPoint 切点
     */
    @AfterReturning(pointcut = "@annotation(controllerLog)", returning = "jsonResult")
    public void doAfterReturning(JoinPoint joinPoint, Log controllerLog, Object jsonResult) {
        handleLog(joinPoint, controllerLog, null, jsonResult);
    }

    /**
     * 拦截异常操作
     *
     * @param joinPoint 切点
     * @param e         异常
     */
    @AfterThrowing(value = "@annotation(controllerLog)", throwing = "e")
    public void doAfterThrowing(JoinPoint joinPoint, Log controllerLog, Exception e) {
        handleLog(joinPoint, controllerLog, e, null);
    }

    private void handleLog(final JoinPoint joinPoint, Log controllerLog, final Exception e, Object jsonResult) {
        try {
            LogInfo logInfo = new LogInfo();
            logInfo.setStatus(Constants.SUCCESS);
            // 请求的地址
            String ip = ServletUtils.getClientIP();
            logInfo.setOperIp(ip);
            logInfo.setOperUrl(ServletUtils.getRequest().getRequestURI());

            logInfo.setUserId(BaseContextHandler.getUid());
            if (e != null) {
                logInfo.setStatus(Constants.FAIL);
                logInfo.setErrorMsg(StringUtils.substring(e.getMessage(), 0, 2000));
            }
            // 设置方法名称
            String className = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            logInfo.setMethod(className + "." + methodName + "()");
            // 设置请求方式
            logInfo.setRequestMethod(ServletUtils.getRequest().getMethod());
            logInfo.setOperTime(LocalDateTime.now());
            // 处理设置注解上的参数
            getControllerMethodDescription(joinPoint, controllerLog, logInfo, jsonResult);
            log.info(logInfo.toString());
            //rocketMQTemplate.sendOneWay(MQKey.TOPIC_LOG + ":" + logTag, logInfo.toString());
        } catch (Exception exp) {
            log.error("前置通知异常", exp);
            // 记录本地异常日志
            //rocketMQTemplate.sendOneWay(MQKey.TOPIC_LOG + ":" + logTag, "前置通知异常》》》" + exp.getMessage());
            exp.printStackTrace();
        }
    }

    /**
     * 获取注解中对方法的描述信息 用于Controller层注解
     *
     * @param log     日志
     * @param operLog 操作日志
     */
    public void getControllerMethodDescription(JoinPoint joinPoint, Log log, LogInfo operLog, Object jsonResult) {
        // 设置action动作
        operLog.setBusinessType(log.businessType().getCode());
        // 设置标题
        operLog.setTitle(log.title());
        Terminal terminal = BaseContextHandler.getTerminal();
        if (terminal != null) {
            operLog.setTerminal(terminal.name());
        }
        operLog.setPlatform(BaseContextHandler.getPlatform());
        operLog.setVersion(BaseContextHandler.getVersion());
        // 是否需要保存request，参数和值
        if (log.isSaveRequestData()) {
            // 获取参数的信息
            setRequestValue(joinPoint, operLog);
        }
        // 是否需要保存response，参数和值
        if (log.isSaveResponseData() && StringUtils.isNotNull(jsonResult)) {
            operLog.setJsonResult(StringUtils.substring(JsonUtils.toJsonString(jsonResult), 0, 2000));
        }
    }

    /**
     * 获取请求的参数，放到log中
     *
     * @param logInfo 操作日志
     */
    private void setRequestValue(JoinPoint joinPoint, LogInfo logInfo) {
        String requestMethod = logInfo.getRequestMethod();
        if (HttpMethod.PUT.name().equals(requestMethod) || HttpMethod.POST.name().equals(requestMethod)) {
            String params = argsArrayToString(joinPoint.getArgs());
            logInfo.setOperParam(StringUtils.substring(params, 0, 2000));
        } else {
            Map<?, ?> paramsMap = (Map<?, ?>) ServletUtils.getRequest().getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
            logInfo.setOperParam(StringUtils.substring(paramsMap.toString(), 0, 2000));
        }
    }

    /**
     * 参数拼装
     */
    private String argsArrayToString(Object[] paramsArray) {
        StringBuilder params = new StringBuilder();
        if (paramsArray != null && paramsArray.length > 0) {
            for (Object o : paramsArray) {
                if (StringUtils.isNotNull(o) && !isFilterObject(o)) {
                    try {
                        params.append(JsonUtils.toJsonString(o)).append(" ");
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return params.toString().trim();
    }

    /**
     * 判断是否需要过滤的对象。
     *
     * @param o 对象信息。
     * @return 如果是需要过滤的对象，则返回true；否则返回false。
     */
    private boolean isFilterObject(final Object o) {
        Class<?> clazz = o.getClass();
        if (clazz.isArray()) {
            return clazz.getComponentType().isAssignableFrom(MultipartFile.class);
        } else if (Collection.class.isAssignableFrom(clazz)) {
            Collection collection = (Collection) o;
            for (Object value : collection) {
                return value instanceof MultipartFile;
            }
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map map = (Map) o;
            for (Object value : map.entrySet()) {
                Map.Entry entry = (Map.Entry) value;
                return entry.getValue() instanceof MultipartFile;
            }
        }
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse
                || o instanceof BindingResult;
    }
}
