package cn.huermao.framework.aspectJ;

import cn.huermao.framework.utils.ServletUtils;
import cn.huermao.framework.utils.StringUtils;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.SecureUtil;
import com.baomidou.lock.LockInfo;
import com.baomidou.lock.LockTemplate;
import cn.huermao.framework.annotation.RepeatSubmit;
import cn.huermao.framework.base.BaseContextHandler;
import cn.huermao.framework.constant.Constants;
import cn.huermao.framework.exception.ServiceException;
import cn.huermao.framework.properties.TokenProperties;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;

/**
 * 防止重复提交
 * 接口入参不要放未重写toString()方法的实体，导致point.getArgs()每次的hash值不一样
 *
 * @author Lion Li
 */
@Slf4j
@Aspect
@Component
public class RepeatSubmitAspect {
    private final TokenProperties tokenProperties;
    private final LockTemplate lockTemplate;

    public RepeatSubmitAspect(TokenProperties tokenProperties, LockTemplate lockTemplate) {
        this.tokenProperties = tokenProperties;
        this.lockTemplate = lockTemplate;
    }

    @Before("@annotation(repeatSubmit)")
    public void doBefore(JoinPoint point, RepeatSubmit repeatSubmit) {
        // 如果注解不为0 则使用注解数值
        long interval = repeatSubmit.timeUnit().toMillis(repeatSubmit.interval());
        if (interval < 1000) {
            throw new ServiceException("重复提交间隔时间不能小于'1'秒");
        }
        HttpServletRequest request = ServletUtils.getRequest();
        String nowParams = StrUtil.join(",", point.getArgs());
        // 唯一值（没有消息头则使用请求地址）
        Long userId = BaseContextHandler.getUid();
        String submitKey = userId == null ? null : userId.toString();
        if (StringUtils.isEmpty(submitKey)) {
            submitKey = request.getHeader(tokenProperties.getHeader());
            if (StringUtils.isEmpty(submitKey)) {
                submitKey = ServletUtils.getClientIP();
            }
        }
        submitKey = submitKey + "_" + request.getRequestURI();
        submitKey = SecureUtil.md5(submitKey + ":" + nowParams);
        // 唯一标识（指定key + 消息头）
        String cacheRepeatKey = Constants.REPEAT_SUBMIT_KEY + submitKey;
        LockInfo lock = lockTemplate.lock(cacheRepeatKey, interval, interval >> 1);
        if (lock == null) {
            throw new ServiceException(repeatSubmit.message());
        }
    }
}
