package cn.huermao.framework.base;

import cn.huermao.framework.enums.ResultCode;
import cn.hutool.json.JSONUtil;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.Accessors;

/**
 * 操作消息提醒
 *
 * @author Lion Li
 */
@ApiModel(value = "接口响应", description = "接口响应结构")
@Data
@NoArgsConstructor
@AllArgsConstructor
@Accessors(chain = true)
public class Result<T> {
    public static final Result<Void> REQUEST_INVALID = new Result<>(ResultCode.REQUEST_INVALID.getCode(), ResultCode.REQUEST_INVALID.getMsg());

    private static final long serialVersionUID = 1L;

    /**
     * 状态码
     */
    @ApiModelProperty(value = "状态码", name = "code")
    private int code;

    /**
     * 状态信息
     */
    @ApiModelProperty(value = "状态信息", name = "msg")
    private String msg;

    /**
     * 返回数据
     */
    @ApiModelProperty(value = "返回数据", name = "data")
    private T data;

    public String toJsonStr() {
        return JSONUtil.toJsonStr(this);
    }

    /**
     * 初始化一个新创建的 Result 对象
     *
     * @param code 状态码
     * @param msg  返回内容
     */
    public Result(int code, String msg) {
        this.code = code;
        this.msg = msg;
    }

    /**
     * 提示信息后直接重定向
     *
     * @param msg 返回内容
     * @return 警告消息
     */
    public static Result<String> redirect(String msg, String path) {
        return new Result<>(ResultCode.REDIRECT.getCode(), msg, path);
    }

    /**
     * 操作成功
     *
     * @return 成功
     */
    public static <T> Result<T> success() {
        return Result.success(ResultCode.SUCCESS.getMsg());
    }

    /**
     * 操作成功
     *
     * @param msg 消息提示
     * @return 成功
     */
    public static <T> Result<T> success(String msg) {
        return Result.success(msg, null);
    }

    /**
     * 操作成功
     *
     * @param data 数据
     * @return 成功
     */
    public static <T> Result<T> success(T data) {
        return Result.success(ResultCode.SUCCESS.getMsg(), data);
    }

    /**
     * 操作成功
     *
     * @param msg  消息提示
     * @param data 数据
     * @return 成功
     */
    public static <T> Result<T> success(String msg, T data) {
        return new Result<>(ResultCode.SUCCESS.getCode(), msg, data);
    }

    /**
     * 操作失败
     *
     * @return 失败
     */
    public static <T> Result<T> fail() {
        return Result.fail(ResultCode.FAIL.getMsg());
    }

    /**
     * 操作失败
     *
     * @param msg 消息提示
     * @return 失败
     */
    public static <T> Result<T> fail(String msg) {
        return Result.fail(msg, null);
    }

    /**
     * 操作失败
     *
     * @param data 数据
     * @return 失败
     */
    public static <T> Result<T> fail(T data) {
        return Result.fail(ResultCode.FAIL.getMsg(), data);
    }

    /**
     * 操作失败
     *
     * @param msg  消息提示
     * @param data 数据
     * @return 失败
     */
    public static <T> Result<T> fail(String msg, T data) {
        return new Result<>(ResultCode.FAIL.getCode(), msg, data);
    }

    /**
     * 您无权限
     *
     * @return 您无权限
     */
    public static <T> Result<T> noPermission() {
        return new Result<>(ResultCode.NO_PERMISSION.getCode(), ResultCode.NO_PERMISSION.getMsg());
    }

    /**
     * 返回正确消息并弹出提示框
     *
     * @param msg 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> successToast(String msg) {
        return Result.successToast(msg, null);
    }

    /**
     * 返回正确消息并弹出提示框
     *
     * @param msg  返回信息
     * @param data 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> successToast(String msg, T data) {
        return new Result<>(ResultCode.SUCCESS_TOAST.getCode(), msg, data);
    }

    /**
     * 返回错误消息并弹出提示框
     *
     * @param msg 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> failToast(String msg) {
        return Result.failToast(msg, null);
    }

    /**
     * 返回错误消息并弹出提示框
     *
     * @param msg  返回信息
     * @param data 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> failToast(String msg, T data) {
        return new Result<>(ResultCode.FAIL_TOAST.getCode(), msg, data);
    }

    /**
     * 返回正确消息并弹模态框提示
     *
     * @param msg 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> successPopup(String msg) {
        return Result.successPopup(msg, null);
    }

    /**
     * 返回正确消息并弹模态框提示
     * 弹框仅有【我知道了】按钮，用于着重提示
     * 可点击按钮后处理相关数据
     *
     * @param msg  返回内容
     * @param data 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> successPopup(String msg, T data) {
        return new Result<>(ResultCode.SUCCESS_POPUP.getCode(), msg, data);
    }

    /**
     * 返回错误消息并弹模态框提示
     * 弹框仅有【我知道了】按钮，用于着重提示
     *
     * @param msg 返回内容
     * @return 警告消息
     */
    public static <T> Result<T> failPopup(String msg) {
        return new Result<>(ResultCode.FAIL_POPUP.getCode(), msg, null);
    }
}
