package cn.huermao.framework.config;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.jsontype.impl.LaissezFaireSubTypeValidator;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;

/**
 * redis配置
 *
 * @author 胡二毛
 */
@Slf4j
@Configuration
@EnableCaching
public class RedisConfig extends CachingConfigurerSupport {
    private final long TTL = 604800L;

    @Bean
    @ConditionalOnMissingBean(RedisTemplate.class)
    public RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<String, Object> template = new RedisTemplate<>();
        template.setConnectionFactory(redisConnectionFactory);
        StringRedisSerializer stringRedisSerializer = new StringRedisSerializer();
        Jackson2JsonRedisSerializer jsonRedisSerializer = this.valueSerializer();

        template.setKeySerializer(stringRedisSerializer);
        template.setValueSerializer(jsonRedisSerializer);
        template.setHashKeySerializer(stringRedisSerializer);
        template.setHashValueSerializer(jsonRedisSerializer);
        return template;
    }

    @Bean
    @ConditionalOnMissingBean(StringRedisTemplate.class)
    public StringRedisTemplate stringRedisTemplate(
            RedisConnectionFactory redisConnectionFactory) {
        StringRedisTemplate template = new StringRedisTemplate();
        template.setConnectionFactory(redisConnectionFactory);
        return template;
    }
//
//    @Bean
//    public DefaultRedisScript<Long> limitScript() {
//        DefaultRedisScript<Long> script = new DefaultRedisScript<>();
//        script.setResultType(Long.class);
//        script.setScriptSource(new ResourceScriptSource(new ClassPathResource("lua/limit.lua")));
//        return script;
//    }

    /*
    org.springframework.cache.interceptor包下的
    自定义key的生成策略,对应@Cacheable中的keyGenerator
    实例对象+方法名+参数名
     */
//    @Bean
//    public KeyGenerator keyGenerator() {
//        /*
//        target调用缓存方法的实例
//        method调用缓存的方法
//        params方法的参数
//         */
//        return (tagert, method, params) -> {
//            StringBuilder sb = new StringBuilder();
//            sb.append(tagert.getClass().getName())
//                    .append(method.getName());
//            for (Object param : params) {
//                sb.append(param.toString());
//            }
//            //返回key
//            return sb.toString();
//        };
//    }

    /*
     * 自定义缓存管理
     */
    @Bean
    public CacheManager cacheManager(RedisConnectionFactory factory) {
        return RedisCacheManager.builder(factory).
                cacheDefaults(redisCacheConfiguration(TTL))//默认缓存策略
                //withInitialCacheConfigurations(RedisCacheConfigurationMap()).//配置不同缓存域,不同过期时间
                //此方法会导致执行完当前impl方法之后再缓存，而不是调用impl时即时缓存。这样就会如果手动改变属性，则将改变的属性缓存
                //.transactionAware()//更新删除上锁。
                .build();


//        // 生成自定义缓存块,redis配置(这个取决于项目中是否有需要在使用@Cacheable缓存数据时,希望有不同的过期时间再决定是否使用)
//        Map<String, RedisCacheConfiguration> customCacheBlockConfig = new LinkedHashMap<>();
//        List<Map<String, Object>> list = CacheNameBlock.toList(); // 从自定义的enum(枚举类)拿到设置缓存块名称及对应的生存时间值
//        list.stream().forEach(item-> {
//            customCacheBlockConfig.put((String)item.get("cacheBlockName"),getRedisCacheConfiguration((Duration)item.get("expirationTime"),key,value)); // 缓存时间
//        });
//
//        // 生成 CacheManager
//        return RedisCacheManager
//                .builder(RedisCacheWriter.nonLockingRedisCacheWriter(connectionFactory)) // 非锁定Redis缓存编写器
//                .cacheDefaults(defaultRedisCacheConfiguration) // 缓存默认值配置
//                .withInitialCacheConfigurations(customCacheBlockConfig) // 自定义缓存默认值配置
//                .build(); // 建立
    }

    /*
     * 配置redis的cache策略
     */
    private RedisCacheConfiguration redisCacheConfiguration(long sec) {
        return RedisCacheConfiguration.defaultCacheConfig()
                //设置key的序列化,采用stringRedisSerializer
                .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(keySerializer()))
                //设置value的序列化，采用Jackson2JsonRedis
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(valueSerializer()))
                //变双冒号为单冒号
                .computePrefixWith(name -> name + ":")
                //设置cache的过期策略
                .entryTtl(Duration.ofSeconds(sec));
        //不缓存null的值
        //disableCachingNullValues();
    }

    /*
    key采用序列化策略
     */
    private RedisSerializer<String> keySerializer() {
        return new StringRedisSerializer();
    }

    /*
     *  value采用序列化策略
     */
    private Jackson2JsonRedisSerializer<Object> valueSerializer() {
        Jackson2JsonRedisSerializer<Object> serializer = new Jackson2JsonRedisSerializer<>(Object.class);
        //序列化所有类包括jdk提供的
        ObjectMapper om = new ObjectMapper();
        //设置序列化的域(属性,方法etc)以及修饰范围,Any包括private,public 默认是public的
        //ALL所有方位,ANY所有修饰符
        om.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        //enableDefaultTyping 原来的方法存在漏洞,2.0后改用如下配置
        //指定输入的类型
        //将类型序列化到属性json字符串中
        om.activateDefaultTyping(LaissezFaireSubTypeValidator.instance,
                ObjectMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.PROPERTY);
        //如果java.time包下Json报错,添加如下两行代码
        om.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        om.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        om.registerModule(new JavaTimeModule());

        serializer.setObjectMapper(om);
        return serializer;
    }
}