package cn.huermao.framework.config;

import cn.huermao.framework.properties.SwaggerProperties;
import cn.huermao.framework.properties.TokenProperties;
import cn.huermao.framework.utils.SpringUtils;
import io.swagger.models.auth.In;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.oas.annotations.EnableOpenApi;
import springfox.documentation.service.*;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.Docket;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;

/**
 * /doc.html	Knife4j提供的文档访问地址
 * /v2/api-docs-ext	Knife4j提供的增强接口地址,自2.0.6版本后删除
 * /swagger-resources	Springfox-Swagger提供的分组接口
 * /v2/api-docs	Springfox-Swagger提供的分组实例详情接口
 * /swagger-ui.html	Springfox-Swagger提供的文档访问地址
 * /swagger-resources/configuration/ui	Springfox-Swagger提供
 * /swagger-resources/configuration/security	Springfox-Swagger提供
 * /favicon.ico   图标隐射 registry.addResourceHandler("/favicon.ico").addResourceLocations("classpath:" + File.separator + "static" + File.separator + "favicon.ico");
 *
 * @author 胡二毛
 */
@Configuration
@EnableOpenApi
@ConditionalOnProperty(prefix = "swagger", name = "enable", havingValue = "true")
public class SwaggerConfig {
    private final SwaggerProperties swaggerProperties;
    private final TokenProperties tokenProperties;

    @Autowired
    public SwaggerConfig(SwaggerProperties swaggerProperties, TokenProperties tokenProperties) {
        this.swaggerProperties = swaggerProperties;
        this.tokenProperties = tokenProperties;
    }

    /**
     * 创建API
     */
    @PostConstruct
    public void createRestApi() {
        List<SwaggerProperties.Groups> groups = swaggerProperties.getGroups();
        if (groups != null && !groups.isEmpty()) {
            for (SwaggerProperties.Groups group : groups) {
                String basePackage = group.getBasePackage();
                Docket docket = new Docket(DocumentationType.OAS_30)
                        .enable(swaggerProperties.getEnable())
                        // 用来创建该API的基本信息，展示在文档的页面中（自定义展示的信息）
                        .apiInfo(apiInfo())
                        // 设置哪些接口暴露给Swagger展示
                        .select()
                        //RequestHandlerSelectors, 配置要扫描接口的方式
                        //any()：扫描所有，项目中的所有接口都会被扫描到
                        //none()：不扫描接口
                        //withClassAnnotation()：扫描注解
                        //withMethodAnnotation()：扫描方法上的注解，比如 withMethodAnnotation(GetMapping.class) 只扫描 @GetMapping 标识的 GET 请求
                        //withClassAnnotation()：扫描类上的注解，比如 withClassAnnotation(Controller.class) 只扫描有 @Controller 注解的类中的接口
                        //basePackage()：扫描指定路径，比如 basePackage(“com.test.controller”) 只扫描 controller 包的接口
                        .apis(RequestHandlerSelectors.basePackage(basePackage))
                        //.paths(PathSelectors.ant("/java1234/**"))  // 匹配 /java1234/**请求路径
                        .paths(PathSelectors.any())
                        .build()
                        .groupName(group.getName())
                        // 设置安全模式，swagger可以设置访问token
                        .securitySchemes(securitySchemes())
                        .securityContexts(securityContexts())
                        .pathMapping(swaggerProperties.getPathMapping());
                String beanName = StringUtils.substringAfterLast(basePackage, ".") + "Docket";
                SpringUtils.registerBean(beanName, docket);
            }
        }
    }

    /**
     * 安全模式，这里指定token通过Authorization头请求头传递
     */
    private List<SecurityScheme> securitySchemes() {
        List<SecurityScheme> apiKeyList = new ArrayList<SecurityScheme>();
        String header = tokenProperties.getHeader();
        apiKeyList.add(new ApiKey(header, header, In.HEADER.toValue()));
        return apiKeyList;
    }

    /**
     * 安全上下文
     */
    private List<SecurityContext> securityContexts() {
        List<SecurityContext> securityContexts = new ArrayList<>();
        securityContexts.add(
                SecurityContext.builder()
                        .securityReferences(defaultAuth())
                        .operationSelector(o -> o.requestMappingPattern().matches("/.*"))
                        .build());
        return securityContexts;
    }

    /**
     * 默认的安全上引用
     */
    private List<SecurityReference> defaultAuth() {
        AuthorizationScope authorizationScope = new AuthorizationScope("global", "accessEverything");
        AuthorizationScope[] authorizationScopes = new AuthorizationScope[1];
        authorizationScopes[0] = authorizationScope;
        List<SecurityReference> securityReferences = new ArrayList<>();
        securityReferences.add(new SecurityReference(tokenProperties.getHeader(), authorizationScopes));
        return securityReferences;
    }

    private ApiInfo apiInfo() {
        Contact contact = null;
        String author = swaggerProperties.getAuthor();
        String url = swaggerProperties.getUrl();
        String email = swaggerProperties.getEmail();
        if (author != null || url != null || email != null) {
            contact = new Contact(author, url, email);
        }
        return new ApiInfo(swaggerProperties.getTitle(), swaggerProperties.getDescription(),
                swaggerProperties.getVersion(), swaggerProperties.getTermsOfServiceUrl(),
                contact, swaggerProperties.getLicense(),
                swaggerProperties.getLicenseUrl(), new ArrayList<>(swaggerProperties.getExtensions()));
    }
}
