package cn.huermao.framework.config.ali;

import cn.huermao.framework.properties.ALiProperties;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.common.utils.BinaryUtil;
import com.aliyun.oss.model.DeleteObjectsRequest;
import com.aliyun.oss.model.DeleteObjectsResult;
import com.aliyun.oss.model.MatchMode;
import com.aliyun.oss.model.PolicyConditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author 胡二毛
 */
@Configuration
@ConditionalOnProperty(prefix = "ali.oss", name = "enable", havingValue = "true")
public class ALiOssConfig {
    private final ALiProperties.Oss aLiOssProperties;
    private final OSS ossClient;
    public static final ExecutorService DELETE_OSS_POOL = new ThreadPoolExecutor(20, 50,
            0L, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(), new ThreadFactoryBuilder()
            .setNameFormat("delete-oss-pool-%d").build(), new ThreadPoolExecutor.CallerRunsPolicy());

    @Autowired
    public ALiOssConfig(ALiProperties aLiProperties) {
        this.aLiOssProperties = aLiProperties.getOss();
        ALiProperties.Oss oss = aLiProperties.getOss();
        this.ossClient = new OSSClientBuilder().build(oss.getEndpoint(), oss.getAccessKeyId(), oss.getAccessKeySecret());
    }

    public Map getOssSignature(String dir) {
        long expireEndTime = System.currentTimeMillis() + aLiOssProperties.getTimeout();
        Date expiration = new Date(expireEndTime);
        PolicyConditions policyConds = new PolicyConditions();
        // PostObject请求最大可支持的文件大小为5 GB，即CONTENT_LENGTH_RANGE为5*1024*1024*1024。
        //设置上传文件的大小限制
        policyConds.addConditionItem(PolicyConditions.COND_CONTENT_LENGTH_RANGE, 0, aLiOssProperties.getMaxLength() * 1024 * 1024);
        //根据参数dir计算的policy，如果和前端uploadFile中参数key的相应字段不一致的话是会报错的
        policyConds.addConditionItem(MatchMode.StartWith, PolicyConditions.COND_KEY, dir);
        //过期时间
        String postPolicy = ossClient.generatePostPolicy(expiration, policyConds);

        byte[] binaryData = postPolicy.getBytes(StandardCharsets.UTF_8);

        String encodedPolicy = BinaryUtil.toBase64String(binaryData);
        String postSignature = ossClient.calculatePostSignature(postPolicy);

        HashMap<String, String> map = new HashMap<>();
        map.put("OSSAccessKeyId", aLiOssProperties.getAccessKeyId());
        map.put("endpoint", aLiOssProperties.getBucketUrl());
        map.put("policy", encodedPolicy);
        map.put("signature", postSignature);
        map.put("dir", dir);
        return map;
    }

    public void deleteOSSFile(String... names) {
        if (names == null || names.length == 0) {
            return;
        }
        List<String> list = new ArrayList<>(Arrays.asList(names));
        this.deleteOSSFile(list);
    }

    public void deleteOSSFile(List<String> nameList) {
        if (nameList == null || nameList.isEmpty()) {
            return;
        }
        String bucketUrl = aLiOssProperties.getBucketUrl();
        nameList.removeIf(i -> i == null || !i.startsWith(bucketUrl));
        if (nameList.isEmpty()) {
            return;
        }
        for (int i = 0; i < nameList.size(); i++) {
            nameList.set(i, nameList.get(i).replace(bucketUrl, ""));
        }
        DELETE_OSS_POOL.submit(() -> {
            // 删除文件。key等同于ObjectName，表示删除OSS文件时需要指定包含文件后缀在内的完整路径，例如abc/efg/123.jpg。
            DeleteObjectsResult deleteObjectsResult = ossClient.deleteObjects(new DeleteObjectsRequest(aLiOssProperties.getBucketName()).withKeys(nameList));
            //deletedObjects表示被删除的文件
            //List<String> deletedObjects = deleteObjectsResult.getDeletedObjects();
            // 单例模式不需要关闭OSSClient。
            //ossClient.shutdown();
        });
    }
}
