package cn.huermao.framework.utils;

import cn.huermao.framework.base.BasePageResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author 胡二毛
 */
@Component
public class MongoPageHelper {

    private static final int FIRST_PAGE_NUM = 1;

    public static final String MONGO_ID = "_id";
    private final MongoTemplate mongoTemplate;

    @Autowired
    public MongoPageHelper(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    /**
     * 分页查询，直接返回集合类型的结果.
     *
     * @see MongoPageHelper#pageQuery(org.springframework.data.mongodb.core.query.Query,
     * java.lang.Class, java.lang.String, java.util.function.Function, java.lang.Integer, java.lang.Integer,
     * java.lang.Long)
     */
    public <T> BasePageResult<T> pageQuery(Query query, Class<T> entityClass, String collectionName, Integer pageNum, Integer pageSize
    ) {
        return pageQuery(query, entityClass, collectionName, Function.identity(), pageNum, pageSize, null);
    }

    /**
     * 分页查询，不考虑条件分页，直接使用skip-limit来分页.
     *
     * @see MongoPageHelper#pageQuery(org.springframework.data.mongodb.core.query.Query,
     * java.lang.Class, java.lang.String, java.util.function.Function, java.lang.Integer, java.lang.Integer,
     * java.lang.Long)
     */
    public <T, R> BasePageResult<R> pageQuery(Query query, Class<T> entityClass, String collectionName, Function<T, R> mapper,
                                              Integer pageNum, Integer pageSize) {
        return pageQuery(query, entityClass, collectionName, mapper, pageNum, pageSize, null);
    }

    /**
     * 分页查询.
     *
     * @param query          Mongo Query对象，构造你自己的查询条件.
     * @param entityClass    Mongo collection定义的entity class，用来确定查询哪个集合.
     * @param collectionName Mongo collection名，用来确定查询哪个集合.
     * @param mapper         映射器，你从db查出来的list的元素类型是entityClass, 如果你想要转换成另一个对象，比如去掉敏感字段等，可以使用mapper来决定如何转换.
     * @param pageNum        当前页.
     * @param pageSize       分页的大小.
     * @param lastId         条件分页参数
     *                       如果不跳页，像朋友圈，微博这样下拉刷新的分页需求，需要传递上一页的最后一条记录的ObjectId。 如果是null，则返回pageNum那一页.
     * @param <T>            collection定义的class类型.
     * @param <R>            最终返回时，展现给页面时的一条记录的类型。
     * @return PageResult，一个封装page信息的对象.
     */
    public <T, R> BasePageResult<R> pageQuery(Query query, Class<T> entityClass, String collectionName, Function<T, R> mapper,
                                              Integer pageNum, Integer pageSize, Long lastId) {
        long total = mongoTemplate.count(query, entityClass, collectionName);
        //总页数
        Integer pages = (int) Math.ceil(total / (double) pageSize);

        BasePageResult<R> basePageResult = new BasePageResult<>();
        basePageResult.setTotal(total);
        basePageResult.setPageNum(pageNum);

        if (pageNum <= 0 || pageNum > pages) {
            return basePageResult;
        }
        if (lastId != null) {
            if (pageNum != FIRST_PAGE_NUM) {
                Criteria criteria = new Criteria();
                criteria.and(MONGO_ID).gt(lastId);
                query.addCriteria(criteria);
            }
        } else {
            int skip = pageSize * (pageNum - 1);
            query.skip(skip);
        }
        query.limit(pageSize);

        List<T> entityList = mongoTemplate
                .find(query.with(Sort.by(Sort.Direction.DESC, MONGO_ID)),
                        entityClass, collectionName);
        basePageResult.setList(entityList.stream().map(mapper).collect(Collectors.toList()));
        return basePageResult;
    }
}