package cn.iisme.framework.common;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.annotation.JSONField;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.JsonProperty;

import cn.iisme.framework.common.Order.Direction;
import cn.iisme.framework.utils.BeanUtils;
import io.swagger.annotations.ApiModelProperty;

/**
 * 分页
 */
@JsonAutoDetect(fieldVisibility = Visibility.ANY, getterVisibility = Visibility.ANY, setterVisibility = Visibility.NONE, isGetterVisibility = Visibility.NONE, creatorVisibility = Visibility.NONE)
@JsonInclude(Include.NON_NULL)
public class Page<T> implements Serializable {

	private static final long serialVersionUID = 1L;

	protected static final Logger LOGGER = LoggerFactory.getLogger(Page.class);

	/** 内容 */
	@JsonProperty
	@ApiModelProperty(value = "内容")
	private final List<T> content = new ArrayList<T>();

	/** 总记录数 */
	@JsonProperty
	@ApiModelProperty(value = "总记录数")
	private long total;

	/** 分页信息 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "分页信息", hidden = true)
	private final Pageable pageable;

	/** 中间段页码数 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "中间段页码数", hidden = true)
	private final int segmentCount;

	/** 是否存在上一页 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "是否存在上一页", hidden = true)
	private final boolean hasPrevious;

	/** 是否存在下一页 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "是否存在下一页", hidden = true)
	private final boolean hasNext;

	/** 是否为首页 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "是否为首页", hidden = true)
	private final boolean isFirst;

	/** 是否为末页 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "是否为末页", hidden = true)
	private final boolean isLast;

	/** 上一页页码 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "上一页页码", hidden = true)
	private final int previousPageNumber;

	/** 下一页页码 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "下一页页码", hidden = true)
	private final int nextPageNumber;

	/** 首页页码 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "首页页码", hidden = true)
	private final int firstPageNumber;

	/** 末页页码 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "末页页码", hidden = true)
	private final int lastPageNumber;

	/** 中间段页码 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "中间段页码", hidden = true)
	private final List<Integer> segment = new ArrayList<Integer>();

	/**
	 * 初始化一个新创建的Page对象
	 */
	public Page() {
		this.total = 0L;
		this.pageable = new Pageable();
		this.segmentCount = 0;
		this.hasPrevious = false;
		this.hasNext = false;
		this.isFirst = false;
		this.isLast = false;
		this.previousPageNumber = 0;
		this.nextPageNumber = 0;
		this.firstPageNumber = 0;
		this.lastPageNumber = 0;
	}

	/**
	 * @param content  内容
	 * @param total    总记录数
	 * @param pageable 分页信息
	 */
	public Page(List<T> content, long total, Pageable pageable) {
		this.content.addAll(content);
		this.total = total;
		this.pageable = pageable;

		Integer pageNumber = this.getPageNumber();
		Integer totalPages = this.getTotalPages();
		Integer segmentCount = this.getSegmentCount();

		if (pageNumber == null || pageNumber < 1) {
			pageNumber = 1;
		}
		if (totalPages == null || totalPages < 1) {
			totalPages = 1;
		}
		if (segmentCount == null || segmentCount < 1) {
			segmentCount = 5;
		}
		boolean hasPrevious = pageNumber > 1;
		boolean hasNext = pageNumber < totalPages;
		boolean isFirst = pageNumber == 1;
		boolean isLast = pageNumber.equals(totalPages);
		int previousPageNumber = pageNumber - 1;
		int nextPageNumber = pageNumber + 1;
		int firstPageNumber = 1;
		int lastPageNumber = totalPages;
		int startSegmentPageNumber = pageNumber - (int) Math.floor((segmentCount - 1) / 2D);
		int endSegmentPageNumber = pageNumber + (int) Math.ceil((segmentCount - 1) / 2D);
		if (startSegmentPageNumber < 1) {
			startSegmentPageNumber = 1;
		}
		if (endSegmentPageNumber > totalPages) {
			endSegmentPageNumber = totalPages;
		}
		for (int i = startSegmentPageNumber; i <= endSegmentPageNumber; i++) {
			this.segment.add(i);
		}
		this.segmentCount = segmentCount;
		this.hasPrevious = hasPrevious;
		this.hasNext = hasNext;
		this.isFirst = isFirst;
		this.isLast = isLast;
		this.previousPageNumber = previousPageNumber;
		this.nextPageNumber = nextPageNumber;
		this.firstPageNumber = firstPageNumber;
		this.lastPageNumber = lastPageNumber;
	}

	/**
	 * 获取页码
	 * 
	 * @return 页码
	 */
	@JsonProperty
	@ApiModelProperty(value = "页码")
	public int getPageNumber() {
		return pageable.getPageNumber();
	}

	/**
	 * 获取每页记录数
	 * 
	 * @return 每页记录数
	 */
	@JsonProperty
	@ApiModelProperty(value = "每页记录数")
	public int getPageSize() {
		return pageable.getPageSize();
	}

	/**
	 * 获取搜索属性
	 * 
	 * @return 搜索属性
	 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "搜索属性", hidden = true)
	public String getSearchProperty() {
		return pageable.getSearchProperty();
	}

	/**
	 * 获取搜索值
	 * 
	 * @return 搜索值
	 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "搜索值", hidden = true)
	public String getSearchValue() {
		return pageable.getSearchValue();
	}

	/**
	 * 获取排序属性
	 * 
	 * @return 排序属性
	 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "排序属性", hidden = true)
	public String getOrderProperty() {
		return pageable.getOrderProperty();
	}

	/**
	 * 获取排序方向
	 * 
	 * @return 排序方向
	 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "排序方向", hidden = true)
	public Direction getOrderDirection() {
		return pageable.getOrderDirection();
	}

	/**
	 * 获取排序
	 * 
	 * @return 排序
	 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "排序", hidden = true)
	public List<Order> getOrders() {
		return pageable.getOrders();
	}

	/**
	 * 获取筛选
	 * 
	 * @return 筛选
	 */
	@JSONField(serialize = false)
	@ApiModelProperty(value = "筛选", hidden = true)
	public List<Filter> getFilters() {
		return pageable.getFilters();
	}

	/**
	 * 获取总页数
	 * 
	 * @return 总页数
	 */
	@JsonProperty
	@ApiModelProperty(value = "总页数")
	public int getTotalPages() {
		return (int) Math.ceil((double) getTotal() / (double) getPageSize());
	}

	/**
	 * 获取内容
	 * 
	 * @return 内容
	 */
	public List<T> getContent() {
		return content;
	}

	/**
	 * 获取总记录数
	 * 
	 * @return 总记录数
	 */
	public long getTotal() {
		return total;
	}

	/**
	 * 获取分页信息
	 * 
	 * @return 分页信息
	 */
	public Pageable getPageable() {
		return pageable;
	}

	public int getSegmentCount() {
		return segmentCount;
	}

	public boolean isHasPrevious() {
		return hasPrevious;
	}

	public boolean isHasNext() {
		return hasNext;
	}

	public boolean getIsFirst() {
		return isFirst;
	}

	public boolean getIsLast() {
		return isLast;
	}

	public int getPreviousPageNumber() {
		return previousPageNumber;
	}

	public int getNextPageNumber() {
		return nextPageNumber;
	}

	public int getFirstPageNumber() {
		return firstPageNumber;
	}

	public int getLastPageNumber() {
		return lastPageNumber;
	}

	public List<Integer> getSegment() {
		return segment;
	}

	/**
	 * 转换列表数据
	 * 
	 * @作者: yangguangjing
	 * @创建时间: 2019年10月17日 下午12:56:14
	 */
	public <M> Page<M> transform(Class<M> clazz) {
		List<M> list = new ArrayList<M>(getContent().size());
		try {
			for (Object object : getContent()) {
				M t = BeanUtils.toBean(object, clazz);
				list.add(t);
			}
		} catch (Exception e) {
			LOGGER.error("", e);
		}
		return new Page<M>(list, total, pageable);
	}

	/**
	 * 逻辑分页
	 * 
	 * @param data     所有数据
	 * @param pageSize 每页数量
	 * @param pageNum  页码
	 * @return 逻辑分页
	 * 
	 * @作者: yangguangjing
	 * @创建时间: 2019年11月30日 下午2:02:51
	 */
	public Page<T> logicalPage(List<T> data, int pageSize, int pageNum) {
		int fromIndex = (pageNum - 1) * pageSize;
		if (fromIndex >= data.size()) {
			return new Page<T>(Collections.emptyList(), 0, pageable);
		}

		int toIndex = pageNum * pageSize;
		if (toIndex >= data.size()) {
			toIndex = data.size();
		}
		return new Page<T>(data.subList(fromIndex, toIndex), data.size(), pageable);
	}

	public void setTotal(long total) {
		this.total = total;
	}

}