package cn.iisme.framework.common;

import java.io.Serializable;

import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.annotation.JsonProperty;

import cn.iisme.framework.exception.IException;
import io.swagger.annotations.ApiModelProperty;

/**
 * 统一返回结果集对象
 *
 * @作者: yangguangjing
 * @创建时间: 2019年11月2日
 */
public class Response<T> implements Serializable {

	private static final long serialVersionUID = 1L;

	public static final String SUCCESS_CODE = "0";
	public static final String SUCCESS_MESSAGE = "操作成功";
	public static final String ERROR_CODE = "-1";
	public static final String ERROR_MESSAGE = "操作失败";

	/** 成功结果 */
	public static final Response<?> SUCCESS_RESULT = Response.success();

	/** 错误结果 */
	public static final Response<?> ERROR_RESULT = Response.error();

	/** 编码 */
	@JsonProperty
	@ApiModelProperty("编码")
	private String code;

	/** 内容 */
	@JsonProperty
	@ApiModelProperty("内容")
	private String message;

	/** 数据 */
	@JsonProperty
	@ApiModelProperty("数据")
	private T data;

	/**
	 * 初始化一个新创建的 Response 对象，使其表示一个空响应。
	 */
	public Response() {
		this(SUCCESS_CODE, SUCCESS_MESSAGE);
	}

	/**
	 * 初始化一个新创建的 AppResponse 对象
	 * 
	 * @param type
	 *            类型
	 * @param message
	 *            内容
	 */
	public Response(String code, String message) {
		this.code = code;
		this.message = message;
	}

	/**
	 * 初始化一个新创建的 AppResponse 对象
	 * 
	 * @param type
	 *            类型
	 * @param message
	 *            内容
	 * @param data
	 *            数据
	 */
	public Response(String code, String message, T data) {
		this.code = code;
		this.message = message;
		this.data = data;
	}

	/**
	 * 返回成功响应
	 * 
	 * @return 成功响应
	 */
	public static <T> Response<?> success() {
		return new Response<T>(SUCCESS_CODE, SUCCESS_MESSAGE);
	}

	/**
	 * 返回成功响应
	 * 
	 * @param message
	 *            内容
	 * 
	 * @return 成功响应
	 */
	public static <T> Response<?> success(String message) {
		return new Response<T>(Response.SUCCESS_CODE, message);
	}

	/**
	 * 返回成功响应
	 * 
	 * @param data
	 *            数据
	 * @return 成功响应
	 */
	public static <T> Response<T> success(T data) {
		return new Response<T>(SUCCESS_CODE, SUCCESS_MESSAGE, data);
	}

	/**
	 * 返回失败响应
	 * 
	 * @return 失败响应
	 */
	public static <T> Response<?> error() {
		return error(ERROR_MESSAGE);
	}

	/**
	 * 返回失败响应
	 * 
	 * @param message
	 *            内容
	 * @return 失败响应
	 */
	public static <T> Response<?> error(String message) {
		return new Response<T>(ERROR_CODE, message);
	}

	/**
	 * 返回失败响应
	 * 
	 * @param code
	 *            编码
	 * @param message
	 *            内容
	 * @return 失败响应
	 */
	public static <T> Response<?> error(String code, String message) {
		return new Response<T>(code, message);
	}

	/**
	 * 返回失败响应
	 * 
	 * @param exception
	 *            异常公共接口
	 * @return 失败响应
	 */
	public static <T> Response<?> error(IException exception) {
		return new Response<T>(exception.getCode(), exception.getMessage());
	}

	/**
	 * 获取编码
	 * 
	 * @return 编码
	 */
	public String getCode() {
		return code;
	}

	/**
	 * 设置编码
	 * 
	 * @param code
	 *            编码
	 */
	public void setCode(String code) {
		this.code = code;
	}

	/**
	 * 获取内容
	 * 
	 * @return 内容
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * 设置内容
	 * 
	 * @param message
	 *            内容
	 */
	public void setMessage(String message) {
		this.message = message;
	}

	/**
	 * 获取数据
	 * 
	 * @return 数据
	 */
	public T getData() {
		return data;
	}

	/**
	 * 设置数据
	 * 
	 * @param data
	 *            数据
	 */
	public Response<T> setData(T data) {
		this.data = data;
		return this;
	}

	/**
	 * 获取是否成功
	 * 
	 * @return 是否成功
	 */
	@JsonProperty
	@ApiModelProperty("是否成功")
	public boolean isSuccessful() {
		return SUCCESS_CODE.equals(code);
	}

	@Override
	public String toString() {
		return JSON.toJSONString(this);
	}
}
