package cn.iisme.framework.utils;

import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * Bean工具类
 * 
 * @author dengyulong
 *
 */
public class BeanUtils {

	protected static final Logger LOGGER = LoggerFactory.getLogger(BeanUtils.class);

	/**
	 * 对象转换 ,将source对象中的所有属性复制到clazz类的实例中并返回
	 * 这里采用fastjson来实现转换操作，原因是先将对象序列化为串，然后在将串反序列化为另一对象，这一过程中不需要考虑类型不匹配的情况 ，
	 * 比如：包装类型的Integer值为null，需要转换到int；又或者强类型转换，两个List泛型类型不一致的情况
	 * 
	 */
	public static <T> T toBean(Object source, Class<T> clazz) {
		if (source == null) {
			return null;
		}
		try {
			String jsonstr = JSON.toJSONString(source);
			return JSON.parseObject(jsonstr, clazz);
		} catch (RuntimeException e) {
			LOGGER.error("", e);
		}
		return null;
	}

	public static <T> T toBean(Object source, Class<T> clazz, String... ignoreProperties) {
		try {
			T t = clazz.newInstance();
			copyProperties(t, source, ignoreProperties);
			return t;
		} catch (InstantiationException e) {
			LOGGER.error("", e);
		} catch (IllegalAccessException e) {
			LOGGER.error("", e);
		}
		return null;
	}

	/**
	 * 复制属性 忽略指定的属性
	 *
	 * @param dest
	 *            目标对象
	 * @param orig
	 *            源对象
	 * @param ignoreProperties
	 *            需要忽略copy的属性
	 * @date 2018年8月24日
	 * @author dengyulong
	 */
	public static void copyProperties(Object dest, Object orig, String[] ignoreProperties) {
		try {
			org.springframework.beans.BeanUtils.copyProperties(orig, dest, ignoreProperties);

		} catch (Exception e) {
			LOGGER.debug("copyProperties() error:" + dest.getClass() + "'", e);
		}
	}

	/**
	 * 转换List
	 */
	public static <T> List<T> transform(List<?> source, Class<T> clazz) {
		List<T> list = new ArrayList<T>(source.size());
		try {
			for (Object object : source) {
				T t = toBean(object, clazz);
				list.add(t);
			}
		} catch (Exception e) {
			LOGGER.error("", e);
		}
		return list;
	}

	public static <T> String beanToString(T value) {
		if (value == null) {
			return null;
		}
		String result = "";
		Class<?> clazz = value.getClass();
		if (clazz == int.class || clazz == Integer.class) {
			result = "" + value;
		} else if (clazz == String.class) {
			result = (String) value;
		} else if (clazz == long.class || clazz == Long.class) {
			result = "" + value;
		} else {
			result = JSON.toJSONString(value);
		}
		return result;
	}

}
