package cn.iisme.framework.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;

public class DateUtils {

	private static final Logger logger = LogUtil.get();

	/** 日期格式【年-月-日 时:分:秒】 */
	public static final String TIME_FORMAT_FULL = "yyyy-MM-dd HH:mm:ss";
	/** 日期格式【年月日时分秒】 */
	public static final String TIME_FORMAT_FULL_SIMP = "yyyyMMddHHmmss";
	/** 日期格式【年-月-日】 */
	public static final String TIME_FORMAT_YMD = "yyyy-MM-dd";
	/** 日期格式【年月日】 */
	public static final String TIME_FORMAT_YMD_SIMP = "yyyyMMdd";
	/** 默认日期格式器 */
	public static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	/** 星期 */
	public static final String[] weekDays = { "星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六" };

	/**
	 * 获取某天的开始时间</br>
	 * </br>
	 * 如2015-04-24 12:00:00的开始时间为</br>
	 * 2015-04-24 00:00:00.000
	 * 
	 * @param date
	 * @return 某天的开始时间
	 */
	public static Date getDayBegin(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	/**
	 * 获取某天的结束时间</br>
	 * </br>
	 * 如2015-04-24 12:00:00的结束时间为</br>
	 * 2015-04-25 00:00:00.000
	 * 
	 * @param date
	 * @return 某天的结束时间
	 */
	public static Date getDayEnd(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(getDayBegin(date));
		calendar.add(Calendar.DATE, 1);
		calendar.add(Calendar.MILLISECOND, -1);
		return calendar.getTime();
	}

	/**
	 * 获取某周的开始时间</br>
	 * </br>
	 * 如2020-06-08本周的开始时间为</br>
	 * 2020-06-08 00:00:00.000
	 * 
	 * @param date
	 * @return 某周的开始时间
	 */
	public static Date getWeekBegin(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		int day_of_week = calendar.get(Calendar.DAY_OF_WEEK) - 1;
		if (day_of_week == 0) {
			day_of_week = 7;
		}
		calendar.add(Calendar.DATE, -day_of_week + 1);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	/**
	 * 获取某周的结束时间</br>
	 * </br>
	 * 如2020-06-08本周的结束时间为</br>
	 * 2020-06-08 00:00:00.000
	 * 
	 * @param date
	 * @return 某月的结束时间
	 */
	public static Date getWeekEnd(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(getDayBegin(date));
		int day_of_week = calendar.get(Calendar.DAY_OF_WEEK) - 1;
		if (day_of_week == 0) {
			day_of_week = 7;
		}
		calendar.add(Calendar.DATE, -day_of_week + 8);
		calendar.add(Calendar.MILLISECOND, -1);
		return calendar.getTime();
	}

	/**
	 * 获取某月的开始时间</br>
	 * </br>
	 * 如2015-04月的开始时间为</br>
	 * 2015-04-01 00:00:00.000
	 * 
	 * @param date
	 * @return 某月的开始时间
	 */
	public static Date getMonthBegin(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.MONTH, 0);
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	/**
	 * 获取某月的结束时间</br>
	 * </br>
	 * 如2015-04月的结束时间为</br>
	 * 2015-05-01 00:00:00.000
	 * 
	 * @param date
	 * @return 某月的结束时间
	 */
	public static Date getMonthEnd(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(getMonthBegin(date));
		calendar.add(Calendar.MONTH, 1);
		calendar.add(Calendar.MILLISECOND, -1);
		return calendar.getTime();
	}

	/**
	 * 获取某年的开始时间</br>
	 * </br>
	 * 如2015年的开始时间为</br>
	 * 2015-01-01 00:00:00.000
	 * 
	 * @param date
	 * @return 某年的开始时间
	 */
	public static Date getYearBegin(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.set(Calendar.MONTH, 0);
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	/**
	 * 获取某年的结束时间</br>
	 * </br>
	 * 如2015年的结束时间为</br>
	 * 2016-01-01 00:00:00.000
	 * 
	 * @param date
	 * @return 某年的结束时间
	 */
	public static Date getYearEnd(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(getYearBegin(date));
		calendar.add(Calendar.YEAR, 1);
		calendar.add(Calendar.MILLISECOND, -1);
		return calendar.getTime();
	}

	/**
	 * 根据日期格式获取开始时间</br>
	 * </br>
	 * 如2015-04-24 12:00:00对应的格式的开始时间为</br>
	 * yyyyMMdd => 2015-04-24 00:00:00.000</br>
	 * yyyyMM => 2015-04-01 00:00:00.000</br>
	 * yyyy => 2015-01-01 00:00:00.000
	 * 
	 * @param date
	 * @return 开始时间
	 */
	public static Date getBeginByFormat(String format, Date date) {
		if (format == null)
			return date;
		switch (format) {
		// 获取年的起始、结束时间
		case "yyyy":
			return DateUtils.getYearBegin(date);
		// 获取月的起始、结束时间
		case "yyyyMM":
			return DateUtils.getMonthBegin(date);
		// 获取日的起始、结束时间
		case "yyyyMMdd":
			return DateUtils.getDayBegin(date);
		default:
			return date;
		}
	}

	/**
	 * 根据日期格式获取结束时间</br>
	 * </br>
	 * 如2015-04-24 12:00:00对应的格式的开始时间为</br>
	 * yyyyMMdd => 2015-04-25 00:00:00.000</br>
	 * yyyyMM => 2015-05-01 00:00:00.000</br>
	 * yyyy => 2016-01-01 00:00:00.000
	 * 
	 * @param date
	 * @return 开始时间
	 */
	public static Date getEndByFormat(String format, Date date) {
		if (format == null)
			return date;
		switch (format) {
		// 获取年的起始、结束时间
		case "yyyy":
			return DateUtils.getYearEnd(date);
		// 获取月的起始、结束时间
		case "yyyyMM":
			return DateUtils.getMonthEnd(date);
		// 获取日的起始、结束时间
		case "yyyyMMdd":
			return DateUtils.getDayEnd(date);
		default:
			return date;
		}
	}

	/**
	 * 日期对象转为日期字符串
	 * 
	 * @param date 日期对象
	 * @return 日期字符串，默认转换为：yyyy-MM-dd HH:mm:ss
	 */
	public static String format(Date date) {
		return sdf.format(date);
	}

	/**
	 * 日期对象转为日期字符串
	 * 
	 * @param date    日期对象
	 * @param pattern 格式
	 * @return 日期字符串
	 */
	public static String format(Date date, String pattern) {
		return new SimpleDateFormat(pattern).format(date);
	}

	/**
	 * 日期字符串转为日期对象
	 * 
	 * @param source 日期字符串
	 * @return 日期对象
	 */
	public static Date parse(String source) {
		try {
			return sdf.parse(source);
		} catch (ParseException e) {
			logger.error("转换日期错误：" + e.getMessage());
			try {
				return new SimpleDateFormat(TIME_FORMAT_YMD).parse(source);
			} catch (ParseException e1) {
				logger.error("转换日期错误：" + e1.getMessage());
			}
		}
		return null;
	}

	/**
	 * 日期字符串转为日期对象
	 * 
	 * @param source  日期字符串
	 * @param pattern 格式
	 * @return 日期对象
	 */
	public static Date parse(String source, String pattern) {
		try {
			return new SimpleDateFormat(pattern).parse(source);
		} catch (ParseException e) {
			logger.error("转换日期错误：" + e.getMessage());
		}
		return null;
	}

	/**
	 * 获取指定日期多少(N)天之前的日期时间</br>
	 * </br>
	 * 如2018-05-09的3天前日期时间为</br>
	 * 2018-05-06 00:00:00.000
	 * 
	 * @param date      指定日期
	 * @param dayBefore 多少天之前
	 * @return N天之前的日期时间
	 */
	public static Date getDayBefore(Date date, Integer dayBefore) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(getDayBegin(date));
		calendar.add(Calendar.DAY_OF_MONTH, -dayBefore);
		return calendar.getTime();
	}

	/**
	 * 获取指定日期多少(N)天之后的日期时间</br>
	 * </br>
	 * 如2018-05-09的3天后日期时间为</br>
	 * 2018-05-12 00:00:00.000
	 * 
	 * @param date     指定日期
	 * @param dayAfter 多少天之后
	 * @return N天之后的日期时间
	 */
	public static Date getDayAfter(Date date, Integer dayAfter) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(getDayBegin(date));
		calendar.add(Calendar.DAY_OF_MONTH, dayAfter);
		return calendar.getTime();
	}

	/**
	 * 增加年
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addYears(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addYears(date, amount);
	}

	/**
	 * 增加月
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addMonths(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addMonths(date, amount);
	}

	/**
	 * 增加星期
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addWeeks(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addWeeks(date, amount);
	}

	/**
	 * 增加天
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addDays(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addDays(date, amount);
	}

	/**
	 * 增加小时
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addHours(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addHours(date, amount);
	}

	/**
	 * 增加分
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addMinutes(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addMinutes(date, amount);
	}

	/**
	 * 增加秒
	 * 
	 * @param date   日期时间
	 * @param amount 数值
	 * @return 增加后的日期时间
	 */
	public static Date addSeconds(final Date date, final int amount) {
		return org.apache.commons.lang3.time.DateUtils.addSeconds(date, amount);
	}

	/**
	 * 获取两个日期时间的相差毫秒数
	 * 
	 * @param begin 开始时间
	 * @param end   结束时间
	 * @return 相差毫秒数
	 */
	public static long betweenMillis(Date begin, Date end) {
		return end.getTime() - begin.getTime();
	}

	/**
	 * 比较两个日期，相等返回0，大于返回1，小于返回-1
	 * 
	 * @param date        第一个日期
	 * @param anotherDate 第二个日期
	 * @return 相等返回0，大于返回1，小于返回-1
	 */
	public static int compare(Date date, Date anotherDate) {
		return date.compareTo(anotherDate);
	}

	/**
	 * 毫秒数转为天数
	 * 
	 * @param millis 毫秒数
	 * @return 天数
	 */
	public static long toDays(long millis) {
		return TimeUnit.MILLISECONDS.toDays(millis);
	}

	/**
	 * 获取星期几
	 * 
	 * @param datetime 日期时间
	 * @return 星期几
	 */
	public static String dateToWeek(Date datetime) {
		Calendar cal = Calendar.getInstance(); // 获得一个日历
		cal.setTime(datetime);
		int w = cal.get(Calendar.DAY_OF_WEEK) - 1; // 指示一个星期中的某天。
		return weekDays[w];
	}

	public static void main(String[] args) {
		System.out.println("本日开始时间：" + DateUtils.format(getDayBegin(new Date())));
		System.out.println("本日结束时间：" + DateUtils.format(getDayEnd(new Date())));
		System.out.println("本周开始时间：" + DateUtils.format(getWeekBegin(new Date())));
		System.out.println("本周结束时间：" + DateUtils.format(getWeekEnd(new Date())));
		System.out.println("本月开始时间：" + DateUtils.format(getMonthBegin(new Date())));
		System.out.println("本月结束时间：" + DateUtils.format(getMonthEnd(new Date())));
		System.out.println("本年开始时间：" + DateUtils.format(getYearBegin(new Date())));
		System.out.println("本年结束时间：" + DateUtils.format(getYearEnd(new Date())));
		System.out.println("3天前日期时间：" + DateUtils.format(getDayBefore(new Date(), 3)));
		System.out.println("3天后日期时间：" + DateUtils.format(getDayAfter(new Date(), 3)));
		System.out.println("今天星期：" + dateToWeek(new Date()));
	}
}
