package cn.shper.tkui.bottombar

import android.content.Context
import android.content.res.TypedArray
import android.graphics.Canvas
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.children
import androidx.core.view.marginStart
import cn.shper.tkui.R
import cn.shper.tkutils.extension.tk_dp2px
import cn.shper.tkutils.extension.tk_getDisplayMetrics

/**
 * Author: Shper
 * Email: me@shper.cn
 * Version: V0.1 2021/11/15
 */
class TKBottomBar @JvmOverloads constructor(context: Context,
                                            attrs: AttributeSet? = null,
                                            defStyleAttr: Int = 0) :
  LinearLayout(context, attrs, defStyleAttr) {

  private var currentItemPosition: Int = -1
  private var onItemSelectedListener: OnItemSelectedListener? = null

  init {
    val typedArray = context.obtainStyledAttributes(attrs, R.styleable.TKBottomBar)
    setupDefaultUI(typedArray)

    typedArray.recycle()
  }

  private fun setupDefaultUI(typedArray: TypedArray) {
    this.orientation = HORIZONTAL
    this.gravity = Gravity.CENTER

    if (!typedArray.hasValue(R.styleable.TKBottomBar_android_background)) {
      this.setBackgroundColor(context.resources.getColor(R.color.tkui_white))
    }
  }

  override fun onFinishInflate() {
    Log.d("Shper", "onFinishInflate - childCount: ${this.childCount}")

    super.onFinishInflate()
    setChildMargin()
  }

  override fun addView(child: View?) {
    super.addView(child)
    setChildMargin()
  }

  override fun addView(child: View?, index: Int) {
    super.addView(child, index)
    setChildMargin()
  }

  override fun addView(child: View?, params: ViewGroup.LayoutParams?) {
    super.addView(child, params)
    setChildMargin()
  }

  override fun addView(child: View?, index: Int, params: ViewGroup.LayoutParams?) {
    super.addView(child, index, params)
    setChildMargin()
  }

  override fun addView(child: View?, width: Int, height: Int) {
    super.addView(child, width, height)
    setChildMargin()
  }

  override fun addViewInLayout(child: View?, index: Int, params: ViewGroup.LayoutParams?): Boolean {
    val result = super.addViewInLayout(child, index, params)
    setChildMargin()
    return result
  }

  override fun addViewInLayout(child: View?,
                               index: Int,
                               params: ViewGroup.LayoutParams?,
                               preventRequestLayout: Boolean): Boolean {
    val result = super.addViewInLayout(child, index, params, preventRequestLayout)
    setChildMargin()
    return result
  }

  private fun setChildMargin() {
    if (this.childCount < 2) {
      return
    }

    // 计算间隔
    val screenWidth = context.tk_getDisplayMetrics().widthPixels
    val childWidth = context.tk_dp2px(70F)
    val spaceWidth = (screenWidth - (childWidth * this.childCount)) / (this.childCount + 1)

    this.children.forEachIndexed { index, view ->
      when (index) {
        0 -> {
          (view.layoutParams as MarginLayoutParams).marginStart = spaceWidth
          (view.layoutParams as MarginLayoutParams).marginEnd = spaceWidth / 2
        }
        this.childCount - 1 -> {
          (view.layoutParams as MarginLayoutParams).marginStart = spaceWidth / 2
          (view.layoutParams as MarginLayoutParams).marginEnd = spaceWidth
        }
        else -> {
          (view.layoutParams as MarginLayoutParams).marginStart = spaceWidth / 2
          (view.layoutParams as MarginLayoutParams).marginEnd = spaceWidth / 2
        }
      }

      view.setOnClickListener { currentItem ->
        val previousItem = getChildAt(currentItemPosition)
        if (previousItem is TKBottomBarItem) {
          previousItem.setSelected(false)
        }

        if (currentItem is TKBottomBarItem) {
          currentItem.setSelected(true)
        }

        onItemSelectedListener?.onItemSelected(currentItem, this.currentItemPosition, index)
        this.currentItemPosition = index
      }
    }
  }

  fun setCurrentItemPosition(position: Int) {
    this.currentItemPosition = position

    val currentItem = getChildAt(currentItemPosition)
    if (currentItem is TKBottomBarItem) {
      currentItem.setSelected(true)
    }
  }

  fun setOnItemSelectedListener(onItemSelectedListener: OnItemSelectedListener?) {
    this.onItemSelectedListener = onItemSelectedListener
  }

  interface OnItemSelectedListener {
    fun onItemSelected(view: View?, previousPosition: Int, currentPosition: Int)
  }


}