package cn.takujo.common_api.util;

import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import cn.takujo.common_api.exception.AESException;

/**
 * AES算法
 * 
 * @author wzx
 *
 */
public final class AESUtil {

	private static final String CHARSET = "UTF-8";
	private static final String KEY_AES = "AES";

	/**
	 * 加密
	 * 
	 * @param data
	 *            需要加密的内容
	 * @param key
	 *            加密秘钥
	 * @return 加密结果
	 * @throws AESException
	 *             加密失败
	 */
	public static String encrypt(String data, String key) throws AESException {
		return doAES(data, key, Cipher.ENCRYPT_MODE);
	}

	/**
	 * 解密
	 * 
	 * @param data
	 *            待解密内容
	 * @param key
	 *            解密密钥
	 * @return 解密结果
	 * @throws AESException
	 *             解密失败
	 */
	public static String decrypt(String data, String key) throws AESException {
		return doAES(data, key, Cipher.DECRYPT_MODE);
	}

	/**
	 * 加解密
	 * 
	 * @param data
	 *            待处理数据
	 * @param password
	 *            密钥
	 * @param mode
	 *            加解密mode
	 * @return 结果
	 * @throws AESException
	 *             处理异常
	 */
	private static String doAES(String data, String key, int mode) throws AESException {
		if ("".equals(data) || "".equals(key)) {
			throw new AESException("aes03", "key或data为空");
		}
		try {
			boolean encrypt = mode == Cipher.ENCRYPT_MODE;
			byte[] content;
			if (encrypt) {
				content = data.getBytes(CHARSET);
			} else {
				content = parseHexStr2Byte(data);
			}
			KeyGenerator kgen = KeyGenerator.getInstance(KEY_AES);
			kgen.init(128, new SecureRandom(key.getBytes()));
			SecretKey secretKey = kgen.generateKey();
			byte[] enCodeFormat = secretKey.getEncoded();
			SecretKeySpec keySpec = new SecretKeySpec(enCodeFormat, KEY_AES);
			Cipher cipher = Cipher.getInstance(KEY_AES);
			cipher.init(mode, keySpec);
			byte[] result = cipher.doFinal(content);
			if (encrypt) {
				return parseByte2HexStr(result);
			} else {
				return new String(result, CHARSET);
			}
		} catch (InvalidKeyException e) {
			throw new AESException("aes01", "无效的key");
		} catch (UnsupportedEncodingException e) {
			throw new AESException("aes01", "不支持的编码格式");
		} catch (NoSuchAlgorithmException e) {
			throw new AESException("aes01", "无匹配的算法");
		} catch (NoSuchPaddingException e) {
			throw new AESException("aes01", "无匹配填充");
		} catch (IllegalBlockSizeException e) {
			throw new AESException("aes01", "非法的块大小");
		} catch (BadPaddingException e) {
			throw new AESException("aes02", "解密失败");
		}
	}

	/**
	 * 将二进制转换成16进制
	 * 
	 * @param buf
	 *            字节数组
	 * @return 16进制字符串
	 */
	public static String parseByte2HexStr(byte buf[]) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < buf.length; i++) {
			String hex = Integer.toHexString(buf[i] & 0xFF);
			if (hex.length() == 1) {
				hex = '0' + hex;
			}
			sb.append(hex.toUpperCase());
		}
		return sb.toString();
	}

	/**
	 * 将16进制转换为二进制
	 * 
	 * @param hexStr
	 *            16进制字符串
	 * @return 二进制字节数组
	 */
	public static byte[] parseHexStr2Byte(String hexStr) {
		if (hexStr.length() < 1) {
			return null;
		}
		byte[] result = new byte[hexStr.length() / 2];
		for (int i = 0; i < hexStr.length() / 2; i++) {
			int high = Integer.parseInt(hexStr.substring(i * 2, i * 2 + 1), 16);
			int low = Integer.parseInt(hexStr.substring(i * 2 + 1, i * 2 + 2), 16);
			result[i] = (byte) (high * 16 + low);
		}
		return result;
	}

}
