package cn.virens.common;

/**
 * GPS 工具类
 * 
 * @author : virens(原作者：lw
 *         http://blog.csdn.net/jane_ybf/article/details/51178547)
 */
public class GPSUtil {
	public static double pi = 3.1415926535897932384626;
	public static double x_pi = 3.14159265358979324 * 3000.0 / 180.0;
	public static double a = 6378245.0;
	public static double ee = 0.00669342162296594323;

	/**
	 * 84 to 火星坐标系 (GCJ-02) World Geodetic System ==&gt; Mars Geodetic System
	 * 
	 * @param  lat 维度
	 * @param  lng 经度
	 * @return     lat/lng
	 */
	public static double[] gps84ToGcj02(double lat, double lng) {
		if (outOfChina(lat, lng)) return new double[] { lat, lng };

		double dLat = transformLat(lng - 105.0, lat - 35.0);
		double dlng = transformLng(lng - 105.0, lat - 35.0);
		double radLat = lat / 180.0 * pi;
		double magic = Math.sin(radLat);

		magic = 1 - ee * magic * magic;
		double sqrtMagic = Math.sqrt(magic);

		dLat = (dLat * 180.0) / ((a * (1 - ee)) / (magic * sqrtMagic) * pi);
		dlng = (dlng * 180.0) / (a / sqrtMagic * Math.cos(radLat) * pi);

		double mgLat = lat + dLat;
		double mglng = lng + dlng;

		return new double[] { mgLat, mglng };
	}

	/**
	 * * 火星坐标系 (GCJ-02) to 84
	 * 
	 * @param  lat 维度
	 * @param  lng 经度
	 * @return     lat/lng
	 */
	public static double[] gcj02ToGps84(double lat, double lng) {
		double[] gps = transform(lat, lng);
		double lngtitude = lng * 2 - gps[1];
		double latitude = lat * 2 - gps[0];

		return new double[] { latitude, lngtitude };
	}

	/**
	 * 火星坐标系 (GCJ-02) 与百度坐标系 (BD-09) 的转换算法 将 GCJ-02 坐标转换成 BD-09 坐标
	 * 
	 * @param  lat 维度
	 * @param  lng 经度
	 * @return     lat/lng
	 */
	public static double[] gcj02ToBd09(double lat, double lng) {
		double x = lng, y = lat;
		double z = Math.sqrt(x * x + y * y) + 0.00002 * Math.sin(y * x_pi);
		double theta = Math.atan2(y, x) + 0.000003 * Math.cos(x * x_pi);
		double templng = z * Math.cos(theta) + 0.0065;
		double tempLat = z * Math.sin(theta) + 0.006;
		double[] gps = { tempLat, templng };

		return gps;
	}

	/**
	 * 火星坐标系 (GCJ-02) 与百度坐标系 (BD-09) 的转换算法 * * 将 BD-09 坐标转换成GCJ-02 坐标
	 * 
	 * @param  lat 维度
	 * @param  lng 经度
	 * @return     lat/lng
	 */
	public static double[] bd09ToGcj02(double lat, double lng) {
		double x = lng - 0.0065, y = lat - 0.006;
		double z = Math.sqrt(x * x + y * y) - 0.00002 * Math.sin(y * x_pi);
		double theta = Math.atan2(y, x) - 0.000003 * Math.cos(x * x_pi);
		double templng = z * Math.cos(theta);
		double tempLat = z * Math.sin(theta);
		double[] gps = { tempLat, templng };

		return gps;
	}

	/**
	 * 将gps84转为bd09
	 * 
	 * @param  lat 维度
	 * @param  lng 经度
	 * @return     lat/lng
	 */
	public static double[] gps84Tobd09(double lat, double lng) {
		double[] gcj02 = gps84ToGcj02(lat, lng);
		double[] bd09 = gcj02ToBd09(gcj02[0], gcj02[1]);

		return bd09;
	}

	/**
	 * 将bd09转为gps84
	 * 
	 * @param  lat 维度
	 * @param  lng 经度
	 * @return     lat/lng
	 */
	public static double[] bd09Togps84(double lat, double lng) {
		double[] gcj02 = bd09ToGcj02(lat, lng);
		double[] gps84 = gcj02ToGps84(gcj02[0], gcj02[1]);
		// 保留小数点后六位
		gps84[0] = retain6(gps84[0]);
		gps84[1] = retain6(gps84[1]);

		return gps84;
	}

	/**
	 * 保留小数点后六位
	 * 
	 * @param  num
	 * @return
	 */
	private static double retain6(double num) {
		return Double.valueOf(String.format("%.6f", num));
	}

	private static double[] transform(double lat, double lng) {
		if (outOfChina(lat, lng)) { return new double[] { lat, lng }; }
		double dLat = transformLat(lng - 105.0, lat - 35.0);
		double dlng = transformLng(lng - 105.0, lat - 35.0);
		double radLat = lat / 180.0 * pi;
		double magic = Math.sin(radLat);
		magic = 1 - ee * magic * magic;
		double sqrtMagic = Math.sqrt(magic);
		dLat = (dLat * 180.0) / ((a * (1 - ee)) / (magic * sqrtMagic) * pi);
		dlng = (dlng * 180.0) / (a / sqrtMagic * Math.cos(radLat) * pi);
		double mgLat = lat + dLat;
		double mglng = lng + dlng;

		return new double[] { mgLat, mglng };
	}

	private static double transformLat(double x, double y) {
		double ret = -100.0 + 2.0 * x + 3.0 * y + 0.2 * y * y + 0.1 * x * y + 0.2 * Math.sqrt(Math.abs(x));
		ret += (20.0 * Math.sin(6.0 * x * pi) + 20.0 * Math.sin(2.0 * x * pi)) * 2.0 / 3.0;
		ret += (20.0 * Math.sin(y * pi) + 40.0 * Math.sin(y / 3.0 * pi)) * 2.0 / 3.0;
		ret += (160.0 * Math.sin(y / 12.0 * pi) + 320 * Math.sin(y * pi / 30.0)) * 2.0 / 3.0;

		return ret;
	}

	private static double transformLng(double x, double y) {
		double ret = 300.0 + x + 2.0 * y + 0.1 * x * x + 0.1 * x * y + 0.1 * Math.sqrt(Math.abs(x));
		ret += (20.0 * Math.sin(6.0 * x * pi) + 20.0 * Math.sin(2.0 * x * pi)) * 2.0 / 3.0;
		ret += (20.0 * Math.sin(x * pi) + 40.0 * Math.sin(x / 3.0 * pi)) * 2.0 / 3.0;
		ret += (150.0 * Math.sin(x / 12.0 * pi) + 300.0 * Math.sin(x / 30.0 * pi)) * 2.0 / 3.0;

		return ret;
	}

	private static boolean outOfChina(double lat, double lng) {
		if (lng < 72.004 || lng > 137.8347) return true;
		if (lat < 0.8293 || lat > 55.8271) return true;

		return false;
	}
}
