package cn.virens.common;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;

/**
 * HttpRequest 工具类
 * 
 * @author : virens
 */
public class RequestUtil {

	/**
	 * 获取request中的所有值
	 * 
	 * @param  request 请求对象
	 * @return         map value
	 */
	public static Map<String, String[]> getParam(HttpServletRequest request) {
		return request.getParameterMap();
	}

	/**
	 * 获取request中的所有值
	 * 
	 * @param  request 请求对象
	 * @return         map value
	 */
	public static Map<String, String> getParam0(HttpServletRequest request) {
		Map<String, String> result = new HashMap<>();

		Map<String, String[]> map = getParam(request);
		for (Entry<String, String[]> entry : map.entrySet()) {
			result.put(entry.getKey(), ArrayUtil.join(entry.getValue(), ","));
		}

		return result;
	}

	/**
	 * 根据默认值返回对应的类型值
	 * 
	 * @param  <T>          class
	 * @param  request      请求对象
	 * @param  columnName   字段名
	 * @param  defaultValue 默认值
	 * @return              T value
	 */
	@SuppressWarnings("unchecked")
	public static <T> T getValue(HttpServletRequest request, String columnName, T defaultValue) {
		if (defaultValue instanceof String) {
			return (T) getValueStr(request, columnName, (String) defaultValue);
		} else if (defaultValue instanceof Integer) {
			return (T) getValueInt(request, columnName, (Integer) defaultValue);
		} else if (defaultValue instanceof Boolean) {
			return (T) getValueBoolean(request, columnName, (Boolean) defaultValue);
		} else if (defaultValue instanceof Float) {
			return (T) getValueFloat(request, columnName, (Float) defaultValue);
		} else if (defaultValue instanceof Long) {
			return (T) getValueLong(request, columnName, (Long) defaultValue);
		} else {
			return defaultValue;
		}
	}

	/**
	 * 返回对应的类型
	 * 
	 * @param  <T>        class
	 * @param  request    请求对象
	 * @param  columnName 字段名
	 * @param  classname  类型
	 * @return            T value
	 */
	@SuppressWarnings("unchecked")
	public static <T> T getValue(HttpServletRequest request, String columnName, Class<T> classname) {
		if (classname.equals(String.class)) {
			return (T) getValueStr(request, columnName);
		} else if (classname.equals(Integer.class)) {
			return (T) getValueInt(request, columnName);
		} else if (classname.equals(Boolean.class)) {
			return (T) getValueBoolean(request, columnName);
		} else if (classname.equals(Float.class)) {
			return (T) getValueFloat(request, columnName);
		} else if (classname.equals(Long.class)) {
			return (T) getValueLong(request, columnName);
		} else {
			return null;
		}
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            String array
	 */
	public static String[] getValue(HttpServletRequest request, String columnName) {
		String[] values = request.getParameterValues(columnName);
		return (values != null) ? values : new String[] {};
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            string value
	 */
	public static String getValueStr(HttpServletRequest request, String columnName) {
		return getValueStr(request, columnName, null);
	}

	/**
	 * 在request中根据name查找值,如找不到就返回默认值
	 * 
	 * @param  request      请求对象
	 * @param  columnName   参数名
	 * @param  defaultValue 默认值
	 * @return              string value
	 */
	public static String getValueStr(HttpServletRequest request, String columnName, String defaultValue) {
		String sValue = request.getParameter(columnName);
		if (sValue == null || sValue.length() == 0) {
			return defaultValue;
		} else {
			return sValue;
		}
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            integer value
	 */
	public static Integer getValueInt(HttpServletRequest request, String columnName) {
		return getValueInt(request, columnName, 0);
	}

	/**
	 * 在request中根据name查找值,如找不到就返回默认值
	 * 
	 * @param  request      请求对象
	 * @param  columnName   参数名
	 * @param  defaultValue 默认值
	 * @return              integer
	 */
	public static Integer getValueInt(HttpServletRequest request, String columnName, Integer defaultValue) {
		return Convert.toInt(getValueStr(request, columnName), defaultValue);
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            long value
	 */
	public static Long getValueLong(HttpServletRequest request, String columnName) {
		return getValueLong(request, columnName, 0L);
	}

	/**
	 * 在request中根据name查找值,如找不到就返回默认值
	 * 
	 * @param  request      请求对象
	 * @param  columnName   参数名
	 * @param  defaultValue 默认值
	 * @return              long value
	 */
	public static Long getValueLong(HttpServletRequest request, String columnName, Long defaultValue) {
		return Convert.toLong(getValueStr(request, columnName), defaultValue);
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            double value
	 */
	public static Double getValueDouble(HttpServletRequest request, String columnName) {
		return getValueDouble(request, columnName, 0D);
	}

	/**
	 * 在request中根据name查找值,如找不到就返回默认值
	 * 
	 * @param  request      请求对象
	 * @param  columnName   参数名
	 * @param  defaultValue 默认值
	 * @return              doulbe value
	 */
	public static Double getValueDouble(HttpServletRequest request, String columnName, Double defaultValue) {
		return Convert.toDouble(getValueStr(request, columnName), defaultValue);
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            float value
	 */
	public static Float getValueFloat(HttpServletRequest request, String columnName) {
		return getValueFloat(request, columnName, 0F);
	}

	/**
	 * 在request中根据name查找值,如找不到就返回默认值
	 * 
	 * @param  request      请求对象
	 * @param  columnName   参数名
	 * @param  defaultValue 默认值
	 * @return              float value
	 */
	public static Float getValueFloat(HttpServletRequest request, String columnName, Float defaultValue) {
		return Convert.toFloat(getValueStr(request, columnName), defaultValue);
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            boolean value
	 */
	public static Boolean getValueBoolean(HttpServletRequest request, String columnName) {
		return getValueBoolean(request, columnName, false);
	}

	/**
	 * 在request中根据name查找值,如找不到就返回默认值
	 * 
	 * @param  request      请求对象
	 * @param  columnName   参数名
	 * @param  defaultValue 默认值
	 * @return              boolean vlaue
	 */
	public static Boolean getValueBoolean(HttpServletRequest request, String columnName, Boolean defaultValue) {
		return Convert.toBool(getValueStr(request, columnName), defaultValue);
	}

	/**
	 * 获取 - 时间
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            Date对象
	 */
	public static Date getValueDate(HttpServletRequest request, String columnName) {
		return getValueDate(request, columnName, DatePattern.NORM_DATETIME_PATTERN, null);
	}

	/**
	 * 获取 - 时间
	 * 
	 * @param  request    请求对象
	 * @param  columnName 变量名
	 * @param  format     时间格式
	 * @return            Date对象,默认当前时间
	 */
	public static Date getValueDate(HttpServletRequest request, String columnName, String format) {
		return getValueDate(request, columnName, format, new Date());
	}

	/**
	 * 获取 - 时间
	 * 
	 * @param  request      请求对象
	 * @param  columnName   变量名
	 * @param  format       时间格式
	 * @param  defaultValue 默认值
	 * @return              Date对象
	 */
	public static Date getValueDate(HttpServletRequest request, String columnName, String format, Date defaultValue) {
		String sValue = getValueStr(request, columnName);
		if (sValue != null) {
			try {
				return DateUtil.parse(sValue, format);
			} catch (Exception e) {
			}
		}

		return defaultValue;
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            string list value
	 */
	public static List<String> getValueList(HttpServletRequest request, String columnName) {
		return CollUtil.newArrayList(getValue(request, columnName));
	}

	/**
	 * 在request中根据name查找值
	 * 
	 * @param  request    请求对象
	 * @param  columnName 参数名
	 * @return            JSONObject value
	 */
	public static JSONObject getValueJSON(HttpServletRequest request, String columnName) {
		return JSON.parseObject(getValueStr(request, columnName));
	}

	/**
	 * 获取请求ContentType
	 * 
	 * @param  request 请求对象
	 * @return         content type
	 */
	public static String getContentType(HttpServletRequest request) {
		return request.getContentType();
	}

	/**
	 * 获取请求HTTP头
	 * 
	 * @param  request 请求对象
	 * @param  header  header's name
	 * @return         header's value
	 */
	public static String getHeader(HttpServletRequest request, String header) {
		return request.getHeader(header);
	}

	/**
	 * 获取请求类型
	 * 
	 * @param  request 请求对象
	 * @return         method type
	 */
	public static String getMethod(HttpServletRequest request) {
		return request.getMethod();
	}

	/**
	 * 获取Session
	 * 
	 * @param  request 请求对象
	 * @return         session object
	 */
	public static HttpSession getSession(HttpServletRequest request) {
		return request.getSession();
	}

	/**
	 * 获取根据名字获取Session值
	 * 
	 * @param  request 请求对象
	 * @param  name    session's name
	 * @return         session's value
	 */
	public static Object getSession(HttpServletRequest request, String name) {
		return getSession(request).getAttribute(name);
	}

	/**
	 * 获取Cookie
	 * 
	 * @param  request 请求对象
	 * @return         cookie array
	 */
	public static Cookie[] getCookies(HttpServletRequest request) {
		return request.getCookies();
	}

	/**
	 * 获取根据名字获取Cookie值
	 * 
	 * @param  request 请求对象
	 * @param  name    cookie's name
	 * @return         cookie's value
	 */
	public static String getCookies(HttpServletRequest request, String name) {
		for (Cookie cookie : getCookies(request)) {
			if (StrUtil.equalsIgnoreCase(name, cookie.getName())) { //
				return cookie.getValue();
			}
		}
		return null;
	}

	/**
	 * 获取请求的协议
	 * 
	 * @param  request 请求对象
	 * @return         request scheme
	 */
	public static String getRequestScheme(HttpServletRequest request) {
		String scheme = request.getHeader("X-Forwarded-Scheme");

		if (StrUtil.isEmpty(scheme)) {
			return request.getScheme();
		} else {
			return scheme;
		}
	}

	/**
	 * 获取本机Host
	 * 
	 * @param  request 请求对象
	 * @return         host
	 */
	public static String getRequestHost(HttpServletRequest request) {
		return request.getHeader("Host");
	}

	/**
	 * 获取 请求地址
	 * 
	 * @param  request 请求对象
	 * @return         request uri
	 */
	public static String getRequestURI(HttpServletRequest request) {
		String scheme = getRequestScheme(request);
		String host = getRequestHost(request);

		return scheme + "://" + host;
	}

	/**
	 * 获取当前请求地址
	 * 
	 * @param  request 请求对象
	 * @return         request url
	 */
	public static String getRequestURL(HttpServletRequest request) {
		String queryString = request.getQueryString();
		String requestURI = request.getRequestURI();

		if (StrUtil.isNotEmpty(queryString)) {
			return requestURI + "?" + queryString;
		} else {
			return requestURI;
		}
	}

	/**
	 * 获取工程路径
	 * 
	 * @param  request 请求对象
	 * @return         dir
	 */
	public static String getRealPath(HttpServletRequest request) {
		return getRealPath(request, "/");
	}

	/**
	 * 根据request获取相对路径在web项目中的完整地址 request请求
	 * 
	 * @param  request  请求对象
	 * @param  filepath 项目中的相对于contextPath的相对路径
	 * @return          项目所在路径 + 相对路径的完整地址
	 */
	public static String getRealPath(HttpServletRequest request, String filepath) {
		return request.getServletContext().getRealPath(filepath);
	}

	/**
	 * 获取远端IP
	 * 
	 * @param  request 请求对象
	 * @return         remote ip
	 */
	public static String getRemoteAddr(HttpServletRequest request) {
		String address = request.getHeader("X-Forwarded-For");

		if (StrUtil.isEmpty(address) || StrUtil.equalsIgnoreCase(address, "unknown")) {
			address = request.getHeader("WL-Proxy-Client-IP");
		}
		if (StrUtil.isEmpty(address) || StrUtil.equalsIgnoreCase(address, "unknown")) {
			address = request.getHeader("Proxy-Client-IP");
		}
		if (StrUtil.isEmpty(address) || StrUtil.equalsIgnoreCase(address, "unknown")) {
			address = request.getHeader("X-Real-IP");
		}
		if (StrUtil.isEmpty(address) || StrUtil.equalsIgnoreCase(address, "unknown")) {
			address = request.getRemoteAddr();
		}

		return StrUtil.subBefore(address, ",", false);
	}
}
