package cn.virens.common;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

/**
 * 序列化工具
 * 
 * @author : virens
 */
public class SerializableUtil {
	/**
	 * 将对象序列化为byte数组
	 * 
	 * @param  object 待序列化对象
	 * @return        byte数组
	 */
	public static byte[] serialize(Object object) {
		ByteArrayOutputStream outputStream = null;
		try {
			// 序列化对象
			serialize(object, (outputStream = new ByteArrayOutputStream()));

			// 返回byte数组
			return outputStream.toByteArray();
		} finally {
			CloseableUtil.close(outputStream);
		}
	}

	/**
	 * 对象序列化
	 * 
	 * @param  object       待序列化对象
	 * @param  outputStream 序列化结果输出流
	 * @return              序列化结果输出流
	 */
	public static OutputStream serialize(Object object, OutputStream outputStream) {
		if (!(object instanceof Serializable)) { throw new IllegalArgumentException("Requires a Serializable payload but received an object of type [" + object.getClass().getName() + "]"); }

		ObjectOutputStream objectOutputStream = null;

		try {
			objectOutputStream = new ObjectOutputStream(outputStream);
			objectOutputStream.writeObject(object);
			objectOutputStream.flush();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			CloseableUtil.close(objectOutputStream);
		}

		return outputStream;
	}

	/**
	 * 反序列化
	 * 
	 * @param  bs    byte数组
	 * @param  clazz 对象类
	 * @param        <T> 对象
	 * @return       反序列化的对象
	 */
	@SuppressWarnings("unchecked")
	public static <T> T deserialize(byte[] bs, Class<T> clazz) {
		Object object = deserialize(bs);

		if (object == null) {
			return null;
		} else if (clazz.isAssignableFrom(object.getClass())) {
			return (T) object;
		} else {
			return null;
		}
	}

	/**
	 * 反序列化
	 * 
	 * @param  bs byte数组
	 * @return    反序列化的对象
	 */
	public static Object deserialize(byte[] bs) {
		ByteArrayInputStream inputStream = null;
		try {
			return deserialize(inputStream = new ByteArrayInputStream(bs));
		} finally {
			CloseableUtil.close(inputStream);
		}
	}

	/**
	 * 反序列化
	 * 
	 * @param  inputStream 输入流
	 * @param  clazz       对象类
	 * @param              <T> 对象
	 * @return             反序列化的对象
	 */
	@SuppressWarnings("unchecked")
	public static <T> T deserialize(InputStream inputStream, Class<T> clazz) {
		Object object = deserialize(inputStream);

		if (object == null) {
			return null;
		} else if (clazz.isAssignableFrom(object.getClass())) {
			return (T) object;
		} else {
			return null;
		}
	}

	/**
	 * 反序列化
	 * 
	 * @param  inputStream 输入流
	 * @return             反序列化的对象
	 */
	public static Object deserialize(InputStream inputStream) {
		ObjectInputStream objectInputStream = null;

		try {
			objectInputStream = new ObjectInputStream(inputStream);

			// 返回反序列化对象
			return objectInputStream.readObject();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} finally {
			CloseableUtil.close(objectInputStream);
		}

		return null;
	}
}
