package cn.virens.common;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.DigestUtil;

public class StringUtil extends StrUtil {

	/**
	 * <p>
	 * Compares given <code>string</code> to a CharSequences vararg of
	 * <code>searchStrings</code>, returning {@code true} if the <code>string</code>
	 * is equal to any of the <code>searchStrings</code>.
	 * </p>
	 *
	 * <pre>
	 * StringUtils.equalsAny(null, (CharSequence[]) null) = false
	 * StringUtils.equalsAny(null, null, null)    = true
	 * StringUtils.equalsAny(null, "abc", "def")  = false
	 * StringUtils.equalsAny("abc", null, "def")  = false
	 * StringUtils.equalsAny("abc", "abc", "def") = true
	 * StringUtils.equalsAny("abc", "ABC", "DEF") = false
	 * </pre>
	 *
	 * @param  string        to compare, may be {@code null}.
	 * @param  searchStrings a vararg of strings, may be {@code null}.
	 * @return               {@code true} if the string is equal (case-sensitive) to
	 *                       any other element of <code>searchStrings</code>;
	 *                       {@code false} if <code>searchStrings</code> is null or
	 *                       contains no matches.
	 * @since                3.5
	 */
	public static boolean equalsAny(final CharSequence string, final CharSequence... searchStrings) {
		if (ArrayUtil.isNotEmpty(searchStrings)) {
			for (final CharSequence next : searchStrings) {
				if (equals(string, next)) { return true; }
			}
		}
		return false;
	}

	/**
	 * <p>
	 * Compares given <code>string</code> to a CharSequences vararg of
	 * <code>searchStrings</code>, returning {@code true} if the <code>string</code>
	 * is equal to any of the <code>searchStrings</code>, ignoring case.
	 * </p>
	 *
	 * <pre>
	 * StringUtils.equalsAnyIgnoreCase(null, (CharSequence[]) null) = false
	 * StringUtils.equalsAnyIgnoreCase(null, null, null)    = true
	 * StringUtils.equalsAnyIgnoreCase(null, "abc", "def")  = false
	 * StringUtils.equalsAnyIgnoreCase("abc", null, "def")  = false
	 * StringUtils.equalsAnyIgnoreCase("abc", "abc", "def") = true
	 * StringUtils.equalsAnyIgnoreCase("abc", "ABC", "DEF") = true
	 * </pre>
	 *
	 * @param  string        to compare, may be {@code null}.
	 * @param  searchStrings a vararg of strings, may be {@code null}.
	 * @return               {@code true} if the string is equal (case-insensitive)
	 *                       to any other element of <code>searchStrings</code>;
	 *                       {@code false} if <code>searchStrings</code> is null or
	 *                       contains no matches.
	 * @since                3.5
	 */
	public static boolean equalsAnyIgnoreCase(final CharSequence string, final CharSequence... searchStrings) {
		if (ArrayUtil.isNotEmpty(searchStrings)) {
			for (final CharSequence next : searchStrings) {
				if (equalsIgnoreCase(string, next)) { return true; }
			}
		}
		return false;
	}

	/**
	 * <p>
	 * Checks if any one of the CharSequences are empty ("") or null.
	 * </p>
	 *
	 * <pre>
	 * StringUtils.isAnyEmpty(null)             = true
	 * StringUtils.isAnyEmpty(null, "foo")      = true
	 * StringUtils.isAnyEmpty("", "bar")        = true
	 * StringUtils.isAnyEmpty("bob", "")        = true
	 * StringUtils.isAnyEmpty("  bob  ", null)  = true
	 * StringUtils.isAnyEmpty(" ", "bar")       = false
	 * StringUtils.isAnyEmpty("foo", "bar")     = false
	 * </pre>
	 *
	 * @param  css the CharSequences to check, may be null or empty
	 * @return     {@code true} if any of the CharSequences are empty or null
	 * @since      3.2
	 */
	public static boolean isAnyEmpty(final CharSequence... css) {
		if (css.length == 0) { return true; }

		for (final CharSequence cs : css) {
			if (isEmpty(cs)) { return true; }
		}

		return false;
	}

	/**
	 * <p>
	 * Checks if none of the CharSequences are empty ("") or null.
	 * </p>
	 *
	 * <pre>
	 * StringUtils.isNoneEmpty(null)             = false
	 * StringUtils.isNoneEmpty(null, "foo")      = false
	 * StringUtils.isNoneEmpty("", "bar")        = false
	 * StringUtils.isNoneEmpty("bob", "")        = false
	 * StringUtils.isNoneEmpty("  bob  ", null)  = false
	 * StringUtils.isNoneEmpty(" ", "bar")       = true
	 * StringUtils.isNoneEmpty("foo", "bar")     = true
	 * </pre>
	 *
	 * @param  css the CharSequences to check, may be null or empty
	 * @return     {@code true} if none of the CharSequences are empty or null
	 * @since      3.2
	 */
	public static boolean isNoneEmpty(final CharSequence... css) {
		return !isAnyEmpty(css);
	}

	/**
	 * 对String字符串进行MD5加密
	 * 
	 * @param  str 原始字符串
	 * @return     MD5_HEX
	 */
	public static String md5(String str) {
		if (isNotEmpty(str)) {
			return DigestUtil.md5Hex(str);
		} else {
			return null;
		}
	}

	/**
	 * 获取字符串长度
	 * 
	 * @param  str 字符串
	 * @return     长度
	 */
	public static int length(String str) {
		return str == null ? 0 : str.length();
	}
}
