package cn.virens.components.poi;

import java.io.File;
import java.io.FileInputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Row.MissingCellPolicy;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import cn.virens.common.CloseableUtil;
import cn.virens.components.poi.common.ReadUtil;
import cn.virens.components.poi.listener.write.WriteCompleteListener;
import cn.virens.components.poi.listener.write.WriteFormatListener;
import cn.virens.components.poi.listener.write.WriteLoadListener;
import cn.virens.components.poi.listener.write.WriteReadyListener;
import cn.virens.components.poi.write.CellTpl;

/**
 * Excel 导出
 * 
 * @author : virens
 */
public class SimpleExcelExported {
	private static final Logger logger = LoggerFactory.getLogger(SimpleExcelExported.class);

	private final Map<String, WriteFormatListener> fmtMap = new HashMap<>(); // 格式转换
	private final Map<String, CellTpl> tplMap = new HashMap<>(); // 整行模版

	private WriteCompleteListener completeListener; // 写入完成监听
	private WriteReadyListener readyListener; // 准备监听
	private WriteLoadListener loadListener; // 写入监听

	private File tplFile; // 模板Excel路径
	private int tplIndex = 0; // 工作蒲序号

	private CellStyle rowStyle; // 行样式
	private int rowDataIndex; // 数据行
	private int rowHeadIndex; // 表头行

	/**
	 * 开始导出(必须先配置模板)
	 * 
	 * @throws Exception 异常
	 */
	public void exported() throws Exception {
		if (tplFile != null && tplFile.exists() && tplFile.isFile()) {
			this.exported0(tplFile);
		} else {
			this.exported0();
		}
	}

	public void addTpl(String key, CellTpl value) {
		this.tplMap.put(key, value);
	}

	public void addTpl(String key, String name, int index) {
		this.tplMap.put(key, new CellTpl(name, index));
	}

	public void addTpl(String key, String name, int index, CellStyle cellStyle) {
		this.tplMap.put(key, new CellTpl(name, index, cellStyle));
	}

	public void addFormat(String key, WriteFormatListener value) {
		this.fmtMap.put(key, value);
	}

	/**
	 * 获取 模板文件
	 * 
	 * @return tplFile
	 */
	public File getTplFile() {
		return tplFile;
	}

	/**
	 * 设置 模板文件
	 * 
	 * @param tplFile 模板文件
	 */
	public void setTplpath(File tplFile) {
		this.tplFile = tplFile;
	}

	/**
	 * 获取 工作蒲序号
	 * 
	 * @return 工作蒲序号
	 */
	public int getTplIndex() {
		return tplIndex;
	}

	/**
	 * 设置 工作蒲序号
	 * 
	 * @param tplIndex 工作蒲序号
	 */
	public void setTplIndex(int tplIndex) {
		this.tplIndex = tplIndex;
	}

	/**
	 * 获取 行样式
	 * 
	 * @return 行样式
	 */
	public CellStyle getRowStyle() {
		return rowStyle;
	}

	public void setRowStyle(CellStyle rowStyle) {
		this.rowStyle = rowStyle;
	}

	public int getRowDataIndex() {
		return rowDataIndex;
	}

	public void setRowDataIndex(int rowDataIndex) {
		this.rowDataIndex = rowDataIndex;
	}

	public int getRowHeadIndex() {
		return rowHeadIndex;
	}

	public void setRowHeadIndex(int rowHeadIndex) {
		this.rowHeadIndex = rowHeadIndex;
	}

	/**
	 * 获取 写入监听
	 * 
	 * @return writeListener
	 */
	public WriteLoadListener getLoadListener() {
		return loadListener;
	}

	/**
	 * 设置 写入监听
	 * 
	 * @param writeListener 写入监听
	 */
	public void setLoadListener(WriteLoadListener writeListener) {
		this.loadListener = writeListener;
	}

	/**
	 * 获取 准备监听
	 * 
	 * @return readyListener
	 */
	public WriteReadyListener getReadyListener() {
		return readyListener;
	}

	/**
	 * 设置 准备监听
	 * 
	 * @param readyListener 准备监听
	 */
	public void setReadyListener(WriteReadyListener readyListener) {
		this.readyListener = readyListener;
	}

	/**
	 * 获取 写入完成监听
	 * 
	 * @return writeCompleteListener
	 */
	public WriteCompleteListener getCompleteListener() {
		return completeListener;
	}

	/**
	 * 设置 写入完成监听
	 * 
	 * @param writeCompleteListener 写入完成监听
	 */
	public void setCompleteListener(WriteCompleteListener writeCompleteListener) {
		this.completeListener = writeCompleteListener;
	}

	private String format(String key, Map<String, Object> map) {
		WriteFormatListener fmt = fmtMap.get(key);
		if (fmt != null) return fmt.apply(map);

		Object val = map.get(key);
		if (val instanceof Date) {
			return DateUtil.formatDateTime((Date) val);
		} else {
			return String.valueOf(val);
		}
	}

	/**
	 * 以新建方式导出数据
	 * 
	 * @throws Exception Exception
	 */
	private void exported0() throws Exception {
		SXSSFWorkbook xssfWorkbook = new SXSSFWorkbook();
		try {
			logger.debug("新建表格进行导出");

			SXSSFSheet sheet = xssfWorkbook.createSheet("Sheet");

			this.writeHead(sheet);
			this.exported1(sheet);

			logger.debug("导出完成!");
			// 写入完成，如果有完成事件，就调用完成事件
			if (completeListener != null) {
				completeListener.accept(xssfWorkbook);
			}
		} finally {
			CloseableUtil.close(xssfWorkbook);
		}
	}

	/**
	 * 以模板方式开始导出数据
	 * 
	 * @param  tplFile   模板路径
	 * @throws Exception Exception
	 */
	private void exported0(File tplFile) throws Exception {
		FileInputStream inputStream = new FileInputStream(tplFile);
		XSSFWorkbook xssfWorkbook = new XSSFWorkbook(inputStream);
		SXSSFWorkbook sxssfWorkbook = new SXSSFWorkbook(xssfWorkbook);
		try {
			logger.debug("使用模板进行导出：" + tplFile);

			this.readTpl(sxssfWorkbook.getXSSFWorkbook().getSheetAt(tplIndex));
			this.exported1(sxssfWorkbook.getSheetAt(tplIndex));

			logger.debug("导出完成!");
			// 写入完成，如果有完成事件，就调用完成事件
			if (completeListener != null) {
				completeListener.accept(sxssfWorkbook);
			}
		} finally {
			CloseableUtil.close(sxssfWorkbook);
			CloseableUtil.close(xssfWorkbook);
			CloseableUtil.close(inputStream);
		}
	}

	/**
	 * 开始导入
	 * 
	 * @param  sheet     工作簿
	 * @throws Exception Exception
	 */
	private void exported1(Sheet sheet) throws Exception {
		Objects.requireNonNull(loadListener, "读取监听为空");

		// 如果有读取工作蒲的监听，则调用，并根据返回值判断是否继续读取
		if (readyListener != null && !readyListener.apply(sheet)) { //
			throw new Exception("Excel未准备");
		}

		int index = 0;// 获取开始行，结束行
		List<Map<String, Object>> datas = null; // 数据源 & 模板源

		logger.debug("开始导出数据...");

		// 循环获取数据
		while (isEmpty(datas = loadListener.apply(index++)) == false) {
			logger.debug("本次将导出{}/{}条数据...", (index - 1), datas.size());

			for (Map<String, Object> map : datas) {
				Row row = sheet.createRow(rowDataIndex++);

				for (Entry<String, CellTpl> entry : tplMap.entrySet()) {
					CellTpl value = entry.getValue();

					Cell cell = createCell(value, row);
					cell.setCellType(CellType.STRING);
					cell.setCellStyle(value.getStyle());
					cell.setCellValue(format(entry.getKey(), map));
				}
			}
		}
	}

	/**
	 * 从模板工作簿中获取模板
	 * 
	 * @param  sheet     工作簿
	 * @return           工作簿
	 * @throws Exception Excel异常
	 */
	private Sheet readTpl(Sheet sheet) throws Exception {
		Assert.isFalse(rowDataIndex > sheet.getLastRowNum(), "模板行不存在");
		Assert.isFalse(rowHeadIndex > sheet.getLastRowNum(), "表头行不存在");

		logger.debug("正在读取表头&模板...");
		// 获取 表头行 & 模板行
		Row headRow = sheet.getRow(rowHeadIndex);
		Row dataRow = sheet.getRow(rowDataIndex);

		// 获取数据行样式
		this.setRowStyle(dataRow.getRowStyle());

		// 读取单元格模板
		int maxCellNum = dataRow.getLastCellNum();
		for (int i = 0; i < maxCellNum; i++) {
			String key = ReadUtil.getString(dataRow, i);
			String name = ReadUtil.getString(headRow, i);
			CellStyle style = ReadUtil.getCellStyle(dataRow, i);

			if (StrUtil.isNotEmpty(key)) {
				this.addTpl(key, name, i, style);
			} else {
				logger.info("当前单元格无效");
			}
		}

		// 移除模板行
		sheet.removeRow(dataRow);

		return sheet;
	}

	/**
	 * 创建表头
	 * 
	 * @param  sheet     工作蒲
	 * @return           Sheet
	 * @throws Exception Exception
	 */
	private Sheet writeHead(Sheet sheet) throws Exception {
		Assert.isFalse(tplMap.isEmpty(), "模板列不存在");

		logger.debug("正在写入表头...");

		// 创建表头行
		Row headRow = sheet.createRow(rowHeadIndex);
		headRow.setRowStyle(rowStyle);

		// 遍历模板列表，并将每列的表头放置到表头行
		for (CellTpl cellTpl : tplMap.values()) {
			Cell cell = createCell(cellTpl, headRow);

			cell.setCellType(CellType.STRING);
			cell.setCellValue(cellTpl.getName());
			cell.setCellStyle(cellTpl.getStyle());
		}

		return sheet;
	}

	/**
	 * 创建 一个单元格
	 * 
	 * @param  cellnum 列号
	 * @param  row     行
	 * @return         单元格
	 */
	private Cell createCell(CellTpl cellnum, Row row) {
		return row.getCell(cellnum.getIndex(), MissingCellPolicy.CREATE_NULL_AS_BLANK);
	}

	private static boolean isEmpty(List<?> datas) {
		return datas == null || datas.isEmpty();
	}

}
